classdef test_sample < matlab.unittest.TestCase
    %Test the method for sampling algebraic components
    
    properties
        three_var_system
    end
    methods(TestMethodSetup)
        function nameFiles(testCase)
            BertiniClean
            
            polysyms x y z
            p = BertiniLab('function_def',[x*z+y; y*z+x], ...
                'variable_group',[x y z]);
            testCase.three_var_system = p.irreducible_decomposition;
        end
    end
    
    methods(Test)
        function testNoArguments(testCase)
            f = @() testCase.three_var_system.sample;
            testCase.verifyError(f,'BertiniLab:sample:minrhs')
        end
        function testOneArgument(testCase)
            % Default: one sample
            component = struct('dim',1,'component_number',0);
            sample_points = testCase.three_var_system.sample(component);
            testCase.verifySize(sample_points,[1 1])
            testCase.verifyEqual(fieldnames(sample_points),{'x'; 'y'; 'z'})
            testCase.verifyClass(sample_points.x,'polysym')
        end
        function testInvalidDimension(testCase)
            % This component does not exist
            component = struct('dim',2,'component_number',0);
            f = @() testCase.three_var_system.sample(component);            
            testCase.verifyError(f,'BertiniLab:sample:invalidNumber')
        end
        function testInvalidComponent(testCase)
            % This component does not exist
            component = struct('dim',1,'component_number',-1);
            f = @() testCase.three_var_system.sample(component);            
            testCase.verifyError(f,'BertiniLab:sample:invalidNumber')
        end
        function testInvalidNumberOfPoints(testCase)
            % This component does not exist
            component = struct('dim',1,'component_number',1);
            n = -1;
            f = @() testCase.three_var_system.sample(component,n);            
            testCase.verifyError(f,'MATLAB:sample:expectedNonnegative')
        end
        function testValidNumberOfPoints(testCase)
            component = struct('dim',1,'component_number',1);
            n = 10;
            component_sample = testCase.three_var_system.sample(component,n);
            testCase.verifyClass(component_sample,'struct')
            testCase.verifySize(component_sample,[1 1])
            testCase.verifyEqual(fieldnames(component_sample),{'x'; 'y'; 'z'})
            testCase.verifySize(component_sample.x,[1 n])
            testCase.verifyClass(component_sample.x,'polysym')
        end
        function testUseWitnessPoint(testCase)
            % Should be able to plug in a witness point
            component = testCase.three_var_system.components(1);
            n = 10;
            component_sample = testCase.three_var_system.sample(component,n);
            testCase.verifyClass(component_sample,'struct')
            testCase.verifySize(component_sample,[1 1])
            testCase.verifyEqual(fieldnames(component_sample),{'x'; 'y'; 'z'})
            testCase.verifySize(component_sample.x,[1 n])
            testCase.verifyClass(component_sample.x,'polysym')
        end
        function testAlternateFileName(testCase)
            component = testCase.three_var_system.components(1);
            n = 10;
            fname = 'alt_sample_name';
            if exist(fname,'file')==2
                delete(fname)
            end
            component_sample = testCase.three_var_system.sample(component,n,fname);
            testCase.verifyEqual(exist(fname,'file'),2)
            delete(fname)
            
            testCase.verifyClass(component_sample,'struct')
            testCase.verifySize(component_sample,[1 1])
            testCase.verifyEqual(fieldnames(component_sample),{'x'; 'y'; 'z'})
            testCase.verifySize(component_sample.x,[1 n])
            testCase.verifyClass(component_sample.x,'polysym')
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end