classdef test_read_witness_data < matlab.unittest.TestCase
    %Test the method for reading witness data
    
    methods(TestMethodSetup)
        function nameFiles(~)
            BertiniClean            
        end
    end
    
    methods(Test)
        function testThreeDegreeOne(testCase)
            % Test with problem that has three solutions of dimension 1 and
            % degree 1.
            %
            % This does not check the correctness of the Bertini
            % calculations, just that the results I saw in the file are
            % transcribed correctly.
            
            config = struct('TrackType',1); % This invokes the decomposition
            polysyms x y z
            ps = BertiniLab('function_def',[x*z+y; y*z+x], ...
                'variable_group',[x y z],'config',config);
            ps = ps.solve;
            components = ps.read_witness_data;
            
            % Check integer values
            testCase.verifySize(components,[1 3])
            testCase.verifyEqual([components.codim],[2 2 2])
            testCase.verifyEqual([components.endpoint_precision],[52 52 52])
            testCase.verifyEqual([components.type],[10 10 10])
            testCase.verifyEqual([components.multiplicity],[1 1 1])
            testCase.verifyEqual([components.component_number],[0 1 2])
            testCase.verifyEqual([components.deflations_needed],[0 0 0])
            testCase.verifyEqual([components.degree],[1 1 1])
    
            % Check data values
            values = [components.endpoint];
            affine_values = double(values(2:4,:)./repmat(values(1,:),3,1));
            affine_values = round(affine_values*1e9)/1e9;
            
            % [x -x 1] solution
            idx1 = find(affine_values(3,:)==1);
            testCase.verifyNotEmpty(idx1)
            testCase.verifyEqual(affine_values(1,idx1),-affine_values(2,idx1),'AbsTol',1e-9)
            
            % [x x -1] solution
            idx2 = find(affine_values(3,:)==-1);
            testCase.verifyNotEmpty(idx2)
            testCase.verifyEqual(affine_values(1,idx2),affine_values(2,idx2),'AbsTol',1e-9)
            
            % [0 0 z] solution
            idx3 = setdiff(1:3,[idx1 idx2]);
            testCase.verifyNotEmpty(idx3)
            testCase.verifyEqual(affine_values(1,idx3),0)
            testCase.verifyEqual(affine_values(2,idx3),0)
        end
        
        function testMixedDegreeAndDimension(testCase)
            polysyms x y z
            S = x^2+y^2+z^2-1;
            T = y-x^2;
            U = z-x^3;
            poly_system = BertiniLab('function_def',[T*S*(x-2); U*S*(y-2); T*U*S*(z-2)], ...
                'variable_group',[x y z]);
            poly_system = poly_system.irreducible_decomposition;
            components = poly_system.read_witness_data;
            
            % Occasionally, Bertini does not find all the components, so
            % here is a crude check to see if they were all found.
            testCase.assertSize(components,[1 6])
            
            % Check integer values
            testCase.verifyEqual([components.codim],[3 2 2 2 2 1])
            testCase.verifyEqual([components.dim],[0 1 1 1 1 2])
            testCase.verifyGreaterThanOrEqual([components.endpoint_precision],52*ones(1,9))
            testCase.verifyEqual([components.type],10*ones(1,9))
            testCase.verifyEqual([components.multiplicity],ones(1,9))
            testCase.verifyEqual([components.component_number],[0 0 1 2 3 3 3 0 0])
            testCase.verifyEqual([components.deflations_needed],zeros(1,9))
            testCase.verifyEqual([components.degree],[1 1 1 1 3 2])
        
        end
    end
    
    methods(TestMethodTeardown)
        function teardown(~)
            BertiniClean
        end
    end
end