classdef test_Newton_iteration < matlab.unittest.TestCase
    %Test the Newton_iteration method.
    
    properties
        alt_input_name
        regular_system
        user_homotopy
        parameter_homotopy
    end
    
    methods(TestMethodSetup)
        function setup(testCase)
            BertiniClean
            
            testCase.alt_input_name = 'alt_input';
            if exist(testCase.alt_input_name,'file')==2
                delete(testCase.alt_input_name)
            end
            
            config = struct('SharpenDigits',20);
            polysyms x y
            circle = x^2 + (y-1)^2 - 1;
            parabola = y-2*x^2;
            testCase.regular_system = BertiniLab('function_def',[circle; parabola], 'variable_group',[x y], ...
                'config',config);
            
            polysyms z H Pi t s
            config = struct('UserHomotopy',1); % User-defined homotopy
            q = polysym('q',[2 1]);
            qvals = [cos(2*Pi*(1-t)); sin(2*Pi*(1-t))];
            testCase.user_homotopy = BertiniLab('config',config,'function_def',z^2-s,'function_name',H, ...
                'variable',z,'pathvariable',t,'parameter',[q qvals],'subfunction', ...
                [s q(1)+1i*q(2)],'starting_points',1);
            
            polysyms x
            p = polysym('a',[1 7]);
            config = struct('ParameterHomotopy',1);
            f = polyval(p,x);
            testCase.parameter_homotopy = BertiniLab('function_def',f,'variable_group',x, ...
                'parameter',p,'config',config);
        end
    end
    
    methods(Test)
        
        % Regular homotopy
        function testRegularEmptyPoint(testCase)
            f = @() testCase.regular_system.Newton_iteration;
            testCase.verifyError(f,'BertiniLab:Newton_iteration:minrhs')
        end
        function testRegularNoPoints(testCase)
            f = @() testCase.regular_system.Newton_iteration([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testNewPointsOnly(testCase)
            newPoints = testCase.regular_system.Newton_iteration([.75, 2; 1e-15 .25].');
            testCase.verifyEqual(exist(fullfile(pwd,'newPoints'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'successFlag'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'newtonResidual'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'CN'),'file'),0)
            testCase.verifySize(newPoints,[2 2])
        end
        function testThreeOutputs(testCase)
            [newPoints,successFlag,NewtonResidual] = testCase.regular_system.Newton_iteration([.75 2; 1e-15 .25].');
            testCase.verifySize(newPoints,[2 2])
            testCase.verifyEqual(successFlag,[true false])
            testCase.verifySize(NewtonResidual,[1 2])
        end
        function testFourOutputs(testCase)
            [newPoints,successFlag,NewtonResidual,conditionNumber] = ...
                testCase.regular_system.Newton_iteration([.75 2; 1e-15 .25].');
            testCase.verifyEqual(exist(fullfile(pwd,'CN'),'file'),2)
            testCase.verifySize(newPoints,[2 2])
            testCase.verifyEqual(successFlag,[true false])
            testCase.verifySize(NewtonResidual,[1 2])
            testCase.verifySize(conditionNumber,[1 2])
        end
        function testRegularChangeInputFileName(testCase)
            newPoints = testCase.regular_system.Newton_iteration([.75, 2; 1e-15 .25].',testCase.alt_input_name);
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
            testCase.verifySize(newPoints,[2 2])
        end
        
        % User homotopy
        function testUserEmptyPoint(testCase)
            f = @() testCase.user_homotopy.Newton_iteration;
            testCase.verifyError(f,'BertiniLab:Newton_iteration:minrhs')
        end
        function testUserNoPoints(testCase)
            f = @() testCase.user_homotopy.Newton_iteration([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testUserNewPointsOnly(testCase)
            newPoints = testCase.user_homotopy.Newton_iteration(1,1);
            testCase.verifyEqual(exist(fullfile(pwd,'newPoints'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'successFlag'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'newtonResidual'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'CN'),'file'),0)
            testCase.verifySize(newPoints,[1 1])
        end
        function testUserSuccessFlag(testCase)
            [newPoints,successFlag] = testCase.user_homotopy.Newton_iteration(1,1);
            testCase.verifySize(newPoints,[1 1])
            testCase.verifyEqual(successFlag,true)
        end
        function testUserThreeOutputs(testCase)
            [newPoints,successFlag,NewtonResidual] = testCase.user_homotopy.Newton_iteration(1,1);
            testCase.verifySize(newPoints,[1 1])
            testCase.verifyEqual(successFlag,true)
            testCase.verifySize(NewtonResidual,[1 1])
        end
        function testUserFourOutputs(testCase)
            [newPoints,successFlag,NewtonResidual,conditionNumber] = ...
                testCase.user_homotopy.Newton_iteration(1,1);
            testCase.verifySize(newPoints,[1 1])
            testCase.verifyTrue(successFlag)
            testCase.verifySize(NewtonResidual,[1 1])
            testCase.verifySize(conditionNumber,[1 1])
        end
        function testUserNoParam(testCase)
            % If no parameter, default of 0 is used.
            actual = testCase.user_homotopy.Newton_iteration(1);
            expected = testCase.user_homotopy.Newton_iteration(1,0);
            testCase.verifyEqual(actual,expected)
        end
        function testUserEmptyParam(testCase)
            actual = testCase.user_homotopy.Newton_iteration(1,[]);
            expected = testCase.user_homotopy.Newton_iteration(1,0);
            testCase.verifyEqual(actual,expected)
        end
        function testUserParamWrongSize(testCase)
            % Extra parameters are ignored
            actual = testCase.user_homotopy.Newton_iteration(1,[1 2]);
            expected = testCase.user_homotopy.Newton_iteration(1,1);
            testCase.verifyEqual(actual,expected)
        end
        function testUserChangeInputFileName(testCase)
            testCase.user_homotopy.Newton_iteration(1,1,testCase.alt_input_name);
            testCase.verifyEqual(exist(fullfile(pwd,'input'),'file'),0)
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
        end
        
        
        % Parameter homotopy
        function testParameterEmptyPoint(testCase)
            f = @() testCase.parameter_homotopy.Newton_iteration;
            testCase.verifyError(f,'BertiniLab:Newton_iteration:minrhs')
        end
        function testParameterNoPoints(testCase)
            f = @() testCase.parameter_homotopy.Newton_iteration([]);
            testCase.verifyError(f,'BertiniLab:struct2mat:wrongLength')
        end
        function testParameterWrongNumberOfParameters(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            f = @() poly_sys.Newton_iteration(1,1);
            testCase.verifyError(f,'BertiniLab:solve:callFailed')
        end
        function testParameterNewPointsOnly(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            p = poly(1:6).';
            x = 2.1;
            newPoints = poly_sys.Newton_iteration(x,p);
            testCase.verifyEqual(exist(fullfile(pwd,'newPoints'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'successFlag'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'newtonResidual'),'file'),2)
            testCase.verifyEqual(exist(fullfile(pwd,'CN'),'file'),0)
            testCase.verifyLessThan(abs(double(newPoints)-2),abs(x-2)) % a little convergence expected
        end
        function testParameterThreeOutputs(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            p = poly(1:6).';
            x = 2.1;
            [newPoints,successFlag,NewtonResidual] = poly_sys.Newton_iteration(x,p);
            testCase.verifyLessThan(abs(double(newPoints)-2),abs(x-2)) % a little convergence expected
            testCase.verifyTrue(successFlag)
            testCase.verifySize(NewtonResidual,[1 1])
        end
        function testParameterFourOutputs(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            p = poly(1:6).';
            x = 2.1;
            [newPoints,successFlag,NewtonResidual,conditionNumber] = ...
                poly_sys.Newton_iteration(x,p);
            testCase.verifyLessThan(abs(double(newPoints)-2),abs(x-2)) % a little convergence expected
            testCase.verifyTrue(successFlag)
            testCase.verifySize(NewtonResidual,[1 1])
            testCase.verifySize(conditionNumber,[1 1])
        end
        function testParameterEmptyParam(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            f = @() poly_sys.Newton_iteration(1,[]);
            testCase.verifyError(f,'BertiniLab:solve:callFailed')
        end
        function testParameterChangeInputFileName(testCase)
            poly_sys = testCase.parameter_homotopy;
            poly_sys = poly_sys.solve;
            poly_sys.config.ParameterHomotopy=2;
            p = poly(1:6).';
            x = 2.1;
            newPoints = poly_sys.Newton_iteration(x,p,testCase.alt_input_name);
            testCase.verifyEqual(exist(testCase.alt_input_name,'file'),2)
            testCase.verifyLessThan(abs(double(newPoints)-2),abs(x-2)) % a little convergence expected
        end
        
    end
    
    methods(TestMethodTeardown)
        function teardown(testCase)
            BertiniClean
            if exist(testCase.alt_input_name,'file')==2
                delete(testCase.alt_input_name)
            end
        end
    end
end

