function [sols,varargout] = findRealRoots(polyFun)
%FINDREALROOTS Find the real, finite, isolated roots of a polynomial
%system.
%
% sols = findRealRoots(polyFun) inputs a set of equations (a cell array of
% strings, polysym or sym object) and returns a structure with the
% solutions for each variable in a field with the name of that variable.
% The variable names are automatically obtained from polyFun.
%
% [sols,poly_system] = ... also returns a BertiniLab object, POLY_SYSTEM,
% with the definition of the problem.
%
% Example:
%   sols = findRealRoots({'x^2-1'; 'y^2-1'});
%
% See also: polysolve, BertiniLab

% Find variables automatically
polyFun = polysym(polyFun);
vars = symvar(polyFun);

% Solve and retrieve finite solutions.
poly_system = BertiniLab('function_def',polyFun,'variable_group',vars);
poly_system = solve(poly_system);
sols = poly_system.match_solutions('real_finite_solutions');

% Convert solutions to double precision
sols = structfun(@double,sols,'UniformOutput',false);
sols = structfun(@real,sols,'UniformOutput',false);

% Warn user if there is a problem.
if poly_system.has_failed_paths || poly_system.has_path_crossings
    results = poly_system.solve_summary;
    istart = strfind(results,'Paths Tracked');
    warning(results(istart:end));
end

% Return BertiniLab object, if desired.
if nargout > 1
    varargout{1} = poly_system;
end