%% Code used in "BertiniLab: A MATLAB interface for solving systems of polynomial equations"
%
% This M-file collects all the code that is displayed in the manuscript,
% "BertiniLab: A MATLAB interface for solving systems of polynomial
% equations", by Daniel J. Bates, Andrew J. Newell and Matthew
% Niemerg (accepted for publication in _Numerical Algorithms_). Running
% |publishCode| creates the page |html/codeUsedInManuscript.html|.

%% Section 5.1: Class polysym
x = polysym('x'); y = polysym('y');
disp(x^2+cos(y))

%%
% Equivalence of different |polysym| definitions:
clear
x = polysym('x'); y = polysym('y'); %#ok<NASGU>
whos

%%
clear
polysyms('x','y')
whos

%%
clear
polysyms x y
whos

%%
% Definition of |polysym| vector:
v = polysym('v',[1 2]); disp(v)

%%
M = polysym('M',2); disp(M)

%%
% Matrix operations have the same syntax as for numeric calculations:
disp(M*v.')

%%
% Numbers can be entered directly or combined with a |polysym| object:
x = polysym(1+1i);
disp(x*(1-1i))

%%
% |polysym| variables can be input to a MATLAB function:
f = @(y,z) z^3 + y^2*z + 1;
polysyms y z
disp(f(cos(y),z))

%%
% Importance of the order of evaluation:
polysyms x
disp(9/16*x)
disp(x*9/16)

%%
% Error results from an attempt to assign a |polysym| object to a component
% of a numeric array:
y = polysym('y',2);
x = zeros(2);
try
    x(1) = y(1); %#ok<NASGU>
catch error
    disp(error.message)
end

%%
% Ways of initializing an array so it is compatible with |polysym|:
x = polysym(0,size(y));
disp(x)

%%
x = y*0;
disp(x)

%% Section 5.2: Class |BertiniLab|

% Solve the system $x^2-1=0$, $y^2-4=0$:
polys = BertiniLab('variable_group',{'x','y'}, ...
                'function_def',{'x^2-1';'y^2-4'});
polys = polys.solve;

%%
% Read solutions and match to variables
sols = polys.match_solutions('real_finite_solutions');

%%
% Convert output to double precision
x = real(double(sols.x));
y = real(double(sols.y));
disp([x(:) y(:)])

%%
% Solve $(29/16)z_1^2 -2z_1z_2 = 0$, $z_2-z_1^2=0$.
polysyms z1 z2
f = @(x,y) [(29/16)*x^3 - 2*x*y; y - x^2];
polys = BertiniLab('function_def',f(z1,z2), ...
'variable_group',[z1 z2]);
polys = polys.solve;
disp(polys.solve_summary)

%%
% Solve again with security level reset:
polys.config = struct('SecurityLevel',1);
polys = polys.solve;
disp(polys.solve_summary)

%%
% Solve using multihomogenous homotopy (needs 5 paths instead of 6):
polysyms z1 z2
f = @(x,y) [(29/16)*x^3 - 2*x*y; y - x^2];
polys = BertiniLab('function_def',f(z1,z2), ...
'variable_group',{z1,z2},'config',struct('SecurityLevel',1));
polys = polys.solve;
disp(polys.solve_summary)

%% Section 6: Customizing |BertiniLab|

% A function:
type('findRealRoots')

%%
% A call to the function:
sols = findRealRoots({'x^2-1'; 'y^2-4'});

%% Section 7: Subclassing and parameter homotopy
anisotropyFcn = @(x,K) 2*[x(1)*(x(2)^2+x(3)^2); x(2)*(x(1)^2+x(3)^2); x(3)*(x(1)^2+x(2)^2) - K*x(3)];
constrFcn = @(x) deal(sum(x.^2)-1,2*x(:));

%%
%We solve this system for a trial value of K:
Kval = 0.2; N = 3;
gradFcn = @(x) anisotropyFcn(x,Kval);
polys = solveWithConstraints(gradFcn,N,constrFcn,'config',struct('SecurityLevel',1));
polys = solve(polys);
disp(polys.solve_summary)

%%
%Now we do the ab initio solve for parameter homotopy:
polysyms K
gradFcn = @(x) anisotropyFcn(x,K);
polys = solveWithConstraints(gradFcn,N,constrFcn,'config', ...
    struct('ParameterHomotopy',1,'SecurityLevel',1),'parameter',K);
% polys = solve(polys,'paramotopy.input');

%%
ia = [];
% Sometimes this method obtains the wrong number of starting points.
while (numel(ia) ~= 5)
    polys = polys.solve('paramotopy.input');
    vars = polys.read_solutions('nonsingular_solutions');
    v = round(double(vars)*1e10);
    v = [sort(v(1:2,:)); v(3:4,:)];
    [~,ia] = unique(abs(v.'),'rows');
end
vars_reduced = vars(:,ia(:));

%%
% An example of a run using parameter homotopy follows:
polys.starting_points = vars_reduced;
polys.config.ParameterHomotopy = 2;
polys.final_parameters = 0.2;
polys = solve(polys,'paramotopy.input');
disp(polys.solve_summary)

%%
% Create the solution curves.
N = 200;
Kval = linspace(-1,1,N);
mag = zeros(3,5,N);
lam = zeros(1,5,N);
problemCases = struct.empty;
nCases = 0;
for ii=1:N
    polys.final_parameters = Kval(ii);
    
    polys = solve(polys,'paramotopy.input');
    sols = polys.read_solutions('raw_solutions');
    
    if polys.has_failed_paths || polys.has_path_crossings
        nCases = nCases+1;
        problemCases(nCases).number = ii;
        problemCases(nCases).diagnosis = polys.solve_summary;
    end
        
    mag(:,:,ii) = sols(1:3,:);
    lam(:,:,ii) = sols(4,:);
end
if ~isempty(problemCases)
    warning('There were some failed paths or path crossings. See problemCases for more information.')
end

%%
% Plot magnetization
map = pmkmp(N,'IsoL');
%map = gray(N);
for ii=1:N
    I = all(abs(imag(mag(:,:,ii)))<1e-12);
    mx = real(mag(1,I,ii)); my = real(mag(2,I,ii)); mz = real(mag(3,I,ii));
    m1 = [mx -mx mx -mx mx -mx mx -mx my -my my -my my -my my -my];
    m2 = [my my -my -my my my -my -my mx mx -mx -mx mx mx -mx -mx];
    m3 = [mz mz mz mz -mz -mz -mz -mz mz mz mz mz -mz -mz -mz -mz];
    line(m1,m2,m3,'Marker','.','LineStyle','none','Color',map(ii,:))
end
colorbar
caxis([min(Kval) max(Kval)])
colormap(map)
view(148,26)
axis off equal

% Add axes and an equator
line([0 0],[0 0],[0 1],'Color','k')
line([0 0],[0 1],[0 0],'Color','k')
line([0 1],[0 0],[0 0],'Color','k')
phi = linspace(0,2*pi,200);
x = cos(phi); y = sin(phi); z = zeros(size(phi));
line(x,y,z,'Color','k')

% Make the colorbar a bit smaller.
h_bar = findobj(gcf,'Tag','Colorbar');
initpos = get(h_bar,'Position');
set(h_bar, ...
   'Position',[initpos(1)+initpos(3)*0.15 initpos(2)+initpos(4)*0.15 ...
      initpos(3)*0.7 initpos(4)*0.7])
  
%% Section 8: Positive-dimensional solutions
%
%In real space there are 16 distinct curves. How many are there in complex
%space? The positive-dimensional components of the solution set for this
%problem can be obtained using the method |irreducible_decomposition|:
m = sym('m',[3 1]);
syms K lambda
E = K*(1-m(3)^2) + m(1)^2*m(2)^2+m(2)^2*m(3)^2+m(3)^2*m(1)^2  ...
+ lambda*(m(1)^2+m(2)^2+m(3)^2-1);
f = [diff(E,m(1)); diff(E,m(2)); diff(E,m(3)); diff(E,lambda)];
polys = BertiniLab('variable_group',[m; lambda; K],'function_def',f);
polys = irreducible_decomposition(polys);
disp(polys.solve_summary)

%%
% There are 10 curves of dimension 1 and degree 1. These correspond to the
% isolated points for a given |K|; when |K| is allowed to vary, the points
% form a 1-dimensional object (in real space, a line). There are also 4
% curves of dimension 1 and degree 4, which we will explore below.

%%
% For each component that Bertini finds, it provides a generic point
% ("witness point") from that set. When |irreducible_decomposition| runs,
% it returns information on the witness points in the property
% |components|. This information includes the dimension of the component
% and its degree. The witness points themselves are in the dependent
% property |witness_points|.
disp(polys.witness_points)

%%
% We can look at just the magnetization components for one of the degree-4
% curves:
idx = find([polys.components.degree]==4);
    
%%
% An example of the witness points for these components follows. There is
% one witness point per degree, so there are four witness points per
% component, arranged in columns.
for ii=1:length(idx)
    disp(double(polys.witness_points(idx(ii)).m))
end

%%
% It is easily confirmed that the degree-four components have the general
% form $(a,0,b)$, $(0,a,b)$, $(c,c,d)$ and $(c,-c,d)$. Their intersections
% with real space are circles.

%%
% BertiniLab also has a method for testing membership of a point. For
% example, the following returns a logical array with one of the degree-1
% components marked true.
ismember = polys.membership(struct('m',[0; 0; 1],'lambda',1,'K',1));
disp(ismember.')

%%
% A given component can also be sampled. A dimension and component number
% can be specified in a structure; in Bertini, the component numbers
% increase from zero, so in BertiniLab the nth element of an array of
% components has Bertini component number n-1. However, one can also simply
% choose one of the elements and pass it to the method |sample|. The
% following returns five points from the 13th component:
npoints = 5;
BertiniLab_component_number = 13;
sample_points = polys.sample(polys.components(BertiniLab_component_number),npoints);

%%
BertiniClean % Delete all the files created by Bertini