%% Printing a Witness Set
% Section 11.1 from _Numerically solving polynomial systems with Bertini_,
% by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese and Charles
% W. Wampler (SIAM 2013).
%
% We begin by generating an irreducible decomposition of the union of a
% line and a circle:
polysyms x y
f = (x-y)*(x^2+y^2-1);
poly_system = BertiniLab('variable_group',[x y],'function_def',f);
poly_system = poly_system.irreducible_decomposition('LineCircle.input');

%%
% Bertini is able to store the witness set and linear slicing equations in
% a form that can be easily reused by Bertini. To do this, set |TrackType|
% to 4. The configuration |MPType| determines the precision to use (0 for
% double; 1 for fixed higher precision set by |Precision|; 2 for adaptive
% precision).
poly_system.config = struct('TrackType',4,'MPType',0);
dim = 1;
component = -2;
file_name = 'witness_points';
linear_system_file = 'linear_system';
poly_system.interactive_choices = {dim,component,file_name,linear_system_file};
poly_system = poly_system.solve('LineCircle.print');

%%
% The interactive dialog looks like this:
results = poly_system.solve_summary;
istart = strfind(results,'*************** Components');
disp(results(istart:end))

%%
% This makes the file |witness_points| with three points in it and
% |linear_system|:
type('linear_system')

%% Computing specific codimensions
%
% The polynomial system
%
% $$ x(x^2-y-z^2) $$
%
% $$ x(x+y+z^2-2)(y-5) $$
%
% $$ x(x^2+x-2)(z-2) $$
%
% has the following irreducible components:
%
% 
% * The $x=0$ plane
% * Two quadratic curves: $(1,y,z)$, where $y+z^2=1$; and $(-2,y,z)$, where
% $y+z^2=4$.
% * Two points $(3,5,2)$ and $(-3,5,2)$.
%
% Suppose we are only interested in irreducible components of dimension at
% least 1. Since the curves are embedded in a three-dimensional space, the
% maximum codimension desired is 2. The following code finds those
% components.
config = struct;
config.MaxCodimension = 2; % for components of codim <= 2
polysyms x y z
f1 = x*(x^2-y-z^2);
f2 = x*(x+y+z^2-2)*(y-5);
f3 = x*(x^2+x-2)*(z-2);
poly_system = BertiniLab('variable_group',[x y z],'function_def',[f1; f2; f3], ...
    'config',config);

poly_system = poly_system.irreducible_decomposition;
results = poly_system.solve_summary;
istart = strfind(results,'************** Decomposition');
disp(results(istart:end))

%%
% Now we will find just the one-dimensional components (codimension 2).
config = struct;
config.SpecificCodimension = 2; % for components of codim = 2
config.WitnessGenType = 1;      % use dimension-by-dimension
poly_system.config = config;

poly_system = poly_system.irreducible_decomposition;

results = poly_system.solve_summary;
istart = strfind(results,'************** Decomposition');
disp(results(istart:end))
