%% Numerical accuracy and function residuals
% Example 1.10 from _Numerically solving polynomial systems
% with Bertini_, by Daniel J. Bates, Jonathan D. Haunstein, Andrew J.
% Sommese and Charles W. Wampler (SIAM 2013).
%
% Solve the polynomial
%
% $$p(z) = z^{10} - 30z^9 + 2, $$
%
% which has large function residuals for the default tolerance.
%
% This example illustrates the use of a function in calls to BertiniLab.

f = @(x) x.^10-30*x.^9+2; % The .^ allows vector inputs
polysyms z
poly_system = BertiniLab('function_def',f(z),'variable_group',z);
poly_system = solve(poly_system);
sols = poly_system.match_solutions('raw_solutions');

%%
% A |polysym| object, |z|, was input to the function to create the Bertini
% call. Now we can use the same function on the numerical results to see
% how close to zero it is.
zsols = double(sols.z);
fprintf('%17s %36s\n','z','f(z)')
fprintf('%15.11f + %15.11fi %15.11f + %15.11fi\n',[real(zsols) imag(zsols) real(f(zsols)) imag(f(zsols))].')

%%
% The book reports that |f(z)| is nowhere near zero for the solution that is
% near 30.  Repeated runs of this example indicate that this seems to be
% the typical behavior, though the value of |f(z)| does vary wildly between
% runs, perhaps due to the random choices made by Bertini during the run.   
%
% A greater accuracy may be obtained for the point using a different
% configuration setting:
poly_system.config = struct('MPType',1,'Precision',100);

%%
% We now evaluate the function at the higher precision.
[~,idx] = min(abs(zsols-30));
points = struct('z',sols.z(idx));
fvals = poly_system.evaluate(points);
disp(fvals)

%%
% We redo the calculations at a higher precision and compare the results:
poly_system.config = struct('FinalTol',1e-19);
poly_system = solve(poly_system);
sols = poly_system.match_solutions('raw_solutions');

zsols = double(sols.z);
[~,idx] = min(abs(zsols-30));
points = sols;
points.z = points.z(:,idx);

poly_system.config = struct('MPType',1,'Precision',100);
fvals = poly_system.evaluate(points);

disp([z sols.z(idx)])
disp([polysym('fval') fvals])
