%% Lagrange points
% Section 5.5.1 from _Numerically solving polynomial systems with Bertini_,
% by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese and Charles W. Wampler (SIAM
% 2013).
%
% Find the Lagrange points for two bodies in circular orbits about their
% common center of mass. These are points where a third, smaller body can
% maintain an equilibrium at a fixed position relative to the larger
% bodies.
%
% The following code solves the system, derived in the book:
%
% $$ \rho_1 + \rho_2 = 1 $$
% $$ 1-w\rho_1=0, \quad \mu - w\rho_2=0 $$
%
% It uses a 3-homogeneous treatment with groups
%
% $$ \{(w), (r_1), (d_{13},d_{23},x,y)\}. $$

warning('This calculation may take several minutes!')

% Note that mu should not be used as a variable name in MATLAB because it
% is a function.
polysyms w r1 x y d13 d23 mu0

% Choose value for the mass ratio
constants = [mu0 polysym(9)];

% The following eliminates r2
r2 = 1-r1;

% f=ma on masses 1 and 2
fma1 = w*r1 - 1;
fma2 = w*r2 - mu0;

% Distances between bodies
dist13 = (x-r1)^2 + y^2 - d13^2;
dist23 = (x+r2)^2 + y^2 - d23^2;

% f = ma on mass 3
a = w*d13^3*d23^3; b1 = mu0*d23^3; b2 = d13^3;
fma3x = a*x + b1*(r1-x) + b2*(-r2-x);
fma3y = a*y + b1*(-y)   + b2*(-y);

config = struct('SecurityLevel',1);
poly_system = BertiniLab('variable_group',{w,r1,[x y d13 d23]}, 'constant',constants, ...
    'function_def',[fma1; fma2; dist13; dist23; fma3x; fma3y],'config',config);
poly_system = solve(poly_system);
results = poly_system.solve_summary;
itrack = strfind(results,'Tracking path');
istart = itrack(end);
disp(results(istart:end))

%%
% Now we make the same calculation using regeneration.

polysystem.config = struct('UseRegeneration',1);
poly_system = solve(poly_system);
results = poly_system.solve_summary;
itrack = strfind(results,'Tracking path');
istart = itrack(end);
disp(results(istart:end))
