classdef (InferiorClasses = {?double}) polysym
    %POLYSYM Polynomial string class
    %
    % Class POLYSYM manipulates strings representing polynomials. It is
    % designed to convert MATLAB functions into input to the program
    % Bertini. It therefore encapsulates the constraints imposed by Bertini
    % on input.
    %
    % Complex numbers are represented in the form 'X+Y*I', where X and Y
    % are polynomial expressions and 'I' is reserved for the square root of
    % -1.
    %
    % Arithmetic operations are overloaded, with some limitations. In
    % particular, POLYSYM objects can only be divided by numerical objects,
    % and is mainly intended for polynomial coefficients. Also, expressions
    % can only be raised to integral powers.
    %
    % In applying POLYSYM, some care must be taken with the order of
    % operations. For example, (9/16)*polysym('x') would first evaluate
    % 9/16 and return a double-precision number. To send Bertini the exact
    % expression, use polysym('x')*9/16. The resulting value will be
    % 'x*9/16'.
    %
    % POLYSYM Properties:
    %   Value -             A string
    %
    % POLYSYM Methods:
    %   cellstr -           Convert to cell array of strings
    %   cos -               Cosine
    %   cross -             Vector cross product
    %   ctranspose -        Non-conjugate transpose
    %   cumprod -           Cumulative product
    %   cumsum -            Cumulative sum
    %   diag -              Diagonal matrices and diagonals of a matrix
    %   diff -              Differences between adjacent terms
    %   eq -                Test for equality
    %   exp -               Exponential function (symbolic only)
    %   disp -              Display polysym object
    %   format_equations -  Create cell array of strings representing equations
    %   format_list -       Format array in comma-delimited string
    %   isempty -           Determine which components are empty
    %   ldivide -           Left array division (.\), scalar divisor only
    %   minus -             Subtract one expression from another.
    %   mldivide -          Left matrix division (\), scalar divisor only
    %   mrdivide -          Right matrix division (/), scalar divisor only
    %   mpower -            Take array power (scalar integer power only)
    %   mtimes -            Matrix multiplication (*)
    %   ne -                Test for inequality
    %   plus -              Add expressions
    %   power -             Take array power (integer powers only)
    %   prod -              Take product of elements
    %   rdivide -           Right array division (./), scalar divisor only
    %   sin -               Sine function (symbolic only)
    %   sort -              Sort array in lexicographic order
    %   sum -               Sum of expressions
    %   sym -               Convert to sym class
    %   times -             Element-by-element multiply (.*)
    %   trace -             Sum of diagonal elements
    %   tril -              Lower triangular part of matrix
    %   triu -              Upper triangular part of matrix
    %   uminus -            Unary minus (-)
    
    % TODO: Modify methods to handle multidimensional array
    
    properties
        Value = '0';
    end
    
    methods
        function obj = polysym(varargin)
            % Constructor for polysym
            %
            % OBJ=POLYSYM(X) inputs X, which can be of class CHAR, POLYSYM,
            % DOUBLE, or SYM; or a CELL array of any of these types.
            % Returns OBJ, an array of POLYSYM objects each of which
            % represents a polynomial expression. If X is a POLYSYM array,
            % OBJ is equal to X.
            %
            % A character argument should be a single string; for multiple
            % strings use a cell array. It can include formatting
            % instructions as in SPRINTF; the default has the form 'A%d_%d'
            % for a matrix.
            %
            % OBJ=POLYSYM(X,sz), where X is CHAR, also inputs SZ, the size
            % of the desired array. If SZ is a scalar, a SZ x SZ array is
            % returned. If SZ has two components, the array has size equal
            % to SIZE(X).
            %
            % Example:
            % sz = [1 3];
            % mag = [polysym('x',sz); polysym('y',sz); polysym('z',sz)];
            
            assert(nargin<3,'MATLAB:TooManyInputs','polysym should have at most 2 arguments.')
            switch nargin
                case 0
                    return
                case 1
                    x = varargin{1};
                    sz = 1;
                case 2
                    x = varargin{1};
                    sz = varargin{2};
            end
            
            if isempty(x) && sz==1
                sz = size(x);
            end
            if prod(sz)==0
                obj = polysym.empty(sz); % keep the number of dimensions the same
                return
            end
            validateattributes(x,{'char','polysym','numeric','sym','cell'},{},'polysym','x',1)
            validateattributes(sz,{'numeric'},{'nonempty','nonnegative','integer'},'polysym','sz',2)
            
            if isa(x,'polysym')
                obj = x;
                
            elseif ischar(x)
                assert(size(x,1)==1,'polysym:polysym:wrongSize', ...
                    'To input an array of strings, please use a cell array.')
                
                % Make sure SZ is interpreted in the standard
                % MATLAB way (e.g., as in ZEROS)
                if isscalar(sz)
                    sz = [sz sz];
                end
                
                % Create indexing for array
                [I,J] = ind2sub(sz,1:prod(sz));
                idx = [I; J];
                allones = all(idx==1,2); %remove singleton dimension
                idx = idx(~allones,:);
                
                % Set up formatting for indices, if any
                formatStr = x;
                if ~any(formatStr=='%')
                    integerFormats = {'%d','_%d'};
                    formatStr = [x,integerFormats{1:size(idx,1)}];
                end
                
                % Create polsym array
                sz = obj.array_dims(sz);
                obj(sz{:}) = polysym;
                for k=1:numel(obj)
                    v = sprintf(formatStr,idx(:,k));
                    v = obj.pi_UC(v); % make sure pi is uppercase ('Pi')
                    v = obj.format_imag(v); % put sqrt(-1) in correct format
                    obj(k).Value = v;
                end
                
            elseif iscell(x)
                S = obj.array_dims(size(x));
                obj(S{:}) = polysym; % preallocate array
                
                % Process each string by running through polysym again
                for k=1:numel(obj)
                    obj(k) = polysym(x{k},sz);
                end
                
            elseif isfloat(x)
                x = repmat(x,sz);
                S = obj.array_dims(size(x));
                obj(S{:}) = polysym; % preallocate array
                
                
                for k=1:numel(obj)
                    str = num2str(x(k),15); % Use default for double precision
                    obj(k) = polysym(str);
                end
                
            elseif isa(x,'sym')
                S = obj.array_dims(size(x));
                obj(S{:}) = polysym; % preallocate array
                
                for k=1:numel(x)
                    % The seemingly redundant use of vpa below is needed to
                    % make the result consistent between the two choices of
                    % symbolic engine, Maple and MuPAD.
                    str = char(vpa(x(k),digits));
                    obj(k) = polysym(str);
                end
                
            else
                error('polysym:polysym:invalidInput', ...
                    'Input is not a valid type.')
            end% switch
        end% polysym
                
        s = cellstr(obj)
        
        s = char_base(p)
        
        s = common_name(p)
        
        obj = cos(obj)

        pc = cross(p1,p2,dim)
        
        p = ctranspose(p)
        
        s = cumprod(p,varargin)
        
        s = cumsum(p,varargin)
        
        p = diag(p,k)
        
        y = diff(p,varargin)
        
        disp(p)
        
        x = double(p)
        
        tf = eq(p1,p2)
                
        p = exp(p)
        
        eqns = format_equations(p1,p2)
        
        str = format_array(p,varargin)
        
        p = horzcat(varargin)
        
        tf = isempty(p)
        
        %         tf = isvariablename(p) %Not quite ready
        %
        %         tf = isnumber(p)
        
        p = ldivide(p1,p2)
        
        p = minus(p1,p2)
        
        p = mldivide(p1,p2)
        
        p = mpower(p1,p2)
        
        p = mrdivide(p1,p2)
        
        p = mtimes(p1,p2)
        
        tf = ne(p1,p2)
        
        p = plus(p1,p2)
        
        p = power(p1,p2)
        
        p = prod(p1,varargin)
        
        array_collection = put_in_arrays(p)
        
        p = rdivide(p1,p2)
        
        [re,im] = real_imag(p)
        
        p = sin(p)
        
        [p,varargout] = sort(p,varargin)
        
        p = sum(p,dim)
        
        vars = symvar(p)
        
        p = times(p1,p2)
        
        p = trace(p)
        
        p = tril(p,varargin)
        
        p = triu(p,varargin)
        
        p = uminus(p)
                
        s = sym(p)
        
        p = vertcat(varargin)
        
        R = vpa(p,d)
        
        y = polyval(p,x)
        
    end% ordinary methods
    
    methods (Hidden)
        p_out = assemble_components(parray)
        
        % These have a somewhat limited capability, so I have hidden them.
        tf = isone(p)
        tf = iszero(p)
    end
    
    methods (Access=protected, Hidden)
        
        function s = checkParens(p)
            % Put parentheses around an expression if needed.
            %
            % S = CHECKPARENS(OBJ) inputs OBJ, a POLYSYM object with one
            % member. Checks whether there are more + and - signs than left
            % parentheses. If so, adds parentheses around the expression.
            % Returns the string S.
            %
            % CHECKPARENS is only intended to be called when a
            % multiplication or division occurs.
            %
            % Example:
            %  x = polysym('x'); y = polysym('y');
            %  x*(y+1)
            %
            %  ans =
            %
            %      'x*(y+1)'
            %
            % See also MTIMES,TIMES,MRDIVIDE,RDIVIDE
            
            s = p.Value;
            nLeftParens = numel(strfind(s,'('));
            nRightParens = numel(strfind(s,')'));
            assert(nLeftParens==nRightParens,'polysym:polysym:unbalanceParens', ...
                    'Expression has unbalanced parentheses.')
            nPlusMinus = numel(strfind(s,'+')) + numel(strfind(s,'-'));
            if nLeftParens < nPlusMinus
                s = ['(',s,')'];
            end
        end% checkParens
        
        
        function [p1,p2] = makeSameSize(p1,p2)
            % Expand OBJ1 and OBJ2 into arrays of same size
            %
            % [OBJ1,OBJ2]=makeSameSize(OBJ1,OBJ2) inputs two variables OBJ1
            % and OBJ1 that may be an array of any number of dimensions. If
            % either OBJ1 or OBJ2 is a scalar, it is expanded to an array
            % of the same size as the other input.
            %
            % MAKESAMESIZE is only intended to be called internally by
            % POLYSYM objects.
            %
            % Example:
            %   p1 = polysym('x'); p2 = polysym('y',3)
            %   p1+p2
            %
            %   ans =
            %
            %       'x+y1'    'x+y2'    'x+y3'
            %
            % See also MINUS, PLUS, TIMES.
            
            % Check sizes of inputs
            if isscalar(p1)
                S = num2cell(size(p2));
                p1 = repmat(p1,[S{:}]);
            end
            if isscalar(p2)
                S = num2cell(size(p1));
                p2 = repmat(p2,[S{:}]);
            end
            
        end% makeSameSize
        
        
    end% protected methods
    
    methods (Static, Hidden)
        s = pi_UC(s)
        [matchstr,remainder] = find_imag_numbers(str)
        [matchstr,remainder] = find_real_numbers(str)
        s = format_imag(s)
        
        function dims = array_dims(x)
%             assert(numel(x)<=2,'polysym:polysym:tooManyDimensions', ...
%                 'POLYSYM does not support > 2 dimensions.')
            dims = num2cell(x);
        end
    end
end% classdef
