function [matchstr,remainder] = find_imag_numbers(str)
%FIND_IMAG_NUMBERS Find imaginary parts of complex numbers in a string.
%
% [matchstr,remainder] = find_imag_numbers(p) takes a string and returns
% all imaginary numbers as a cell array of strings in MATCHSTR. The rest of
% the string is returned in REMAINDER.
%
% Imaginary numbers have the form A*I (spaces not important), where A is a
% real number in decimal or scientific notation. Examples of acceptable
% numbers are '3.14159*I', '3 *I', '.3 * I', '1.4e-10*I', '- 1.4';
% unacceptable numbers include '1e*I' and 'e10*I'. This function is only
% intended for use by polysym methods that should check the remainder for
% anomalies.
%
% Any white space in the matches or the remainder is removed.
%
% See also: find_real_number, write_number
% 
% Example:
%   str = 1e-24 - 10 * I;
%   [matchstr,remainder] = find_imag_numbers(str);

e1 = '[+-]?\s*+';                       % optional sign in front
e2 = '(?>\d++\.?+\d*+|\d*+\.?+\d++)';   % digits with optional decimal
e3 = '(e[\+\-]?\d+)?';                  % optional exponent
e4 = '\s*\*\s*I';                       % multiplied by I (square root of -1)
pl = '\(*';                             % possibly preceded by parentheses
pr = '\)*';                             % possible followed by parentheses
expr = [e1,pl,e1,e2,e3,pr,e4];

% This search captures the *I so it is not part of the remainder.
[matchstr,remainder] = regexpi(str, expr, 'match','split');

remainder = deblank(remainder);
remainder = [remainder{:}];
    
for ii=1:numel(matchstr)
    str = regexpi(matchstr{ii},[e1,e2,e3],'match'); % Get rid of *I and parentheses
    str = str{1};
    matchstr{ii} = str(~isspace(str));    % Remove white space
end
