function obj = solve(obj,varargin)
%Solve Solve polynomial system
%
% obj = solve(obj) inputs a BertiniLab object with a
% function to solve and configuration settings; returns the
% object with the solutions assigned to the properties.
%
% obj = solve(obj,input_file_name) gives the name of the input file read by
% Bertini as a string (Default: 'input').
%
% obj = solve(obj, ..., interactive_choices) inputs either a cell
% array of strings or a file name. The file would include one response per
% line to the questions Bertini would ask in an interactive session. The
% cell array gives each response as a separate string.
%
% Example:
%  x = polysym('x'); y = polysym('y');
%  fvals = [x^2-1; x+y-1];
%  fname=polysym({'f','g'});
%
%  obj = BertiniLab('function_def',[fname(:) fvals(:)],'variable_group',[x y]);
%  obj = solve(obj);
%
% See also: readSolutions

% Create input file, start file, interactive choice file and final
% parameters file (as necessary).
obj = obj.make_input_file(varargin{:});
start_file_name = obj.make_data_file(obj.starting_points,'start');
interactive_choices = obj.make_interactive_choices(varargin{2:end});
obj.make_param_file(obj.final_parameters,'final_parameters');
obj.make_data_file(obj.member_points,'member_points');

% Determine whether this run is interactive.
configOptions = fields(obj.config);
iSharpen = find(strcmpi('SharpenOnly',configOptions));
iTrack = find(strcmpi('TrackType',configOptions));
endpointSharpening = ~isempty(iSharpen) && obj.config.(configOptions{iSharpen})==1;
componentSampling = ~isempty(iTrack) && obj.config.(configOptions{iTrack})==2;
witnessSetPrinting = ~isempty(iTrack) && obj.config.(configOptions{iTrack})==4;
isInteractive = endpointSharpening || componentSampling || witnessSetPrinting;

% Use try/catch to make sure program returns to original folder before exiting.
try
    oldFolder = cd(obj.io_folder);
    
    if isInteractive && isempty(interactive_choices)
        system([obj.bertini,' ',obj.input_file_name,' ',start_file_name])
    else
        [status,result] = system([obj.bertini,' ',obj.input_file_name, ...
            ' ',start_file_name,' ',interactive_choices],obj.echo_flag);
        assert(status==0,'BertiniLab:solve:callFailed', ...
            sprintf('Call to Bertini resulted in an error:\n %s\n',result))
        obj.solve_summary = result;
    end
catch err
    cd(oldFolder)
    rethrow(err)
end
cd(oldFolder)

% Flag path crossings and failed paths.
obj.has_failed_paths = ~isempty(strfind(obj.solve_summary,'failed_paths'));
obj.has_path_crossings = ~isempty(strfind(obj.solve_summary,'path crossing'));

