function solution_info = read_raw_data(poly_sys)
%READ_RAW_DATA Read raw_data file (solutions with diagnostics)
%
% solution_info = poly_sys.read_raw_data reads the file raw_data, containing
% solutions and diagnostic information, and returns the data in the
% structure array SOLUTION_INFO.
%
% The fields in SOLUTION_INFO are:
%   path_number -                   integer (counts up from 0)
%   max_precision -                 integer, precision used for the endpoint
%   homogeneous_coordinates -       polysym array representing a complex solution
%   function_residual -             double precision
%   condition_number -              double precision, condition number of the Jacobian matrix
%   final_Newton_residual -         double precision
%   final_pathvariable_value -      double precision
%   last_difference -               double precision, difference between last two estimates at t=0
%   pathvariable_of_first_precision_increase -  double precision
%   endgame_cycle_number -          integer, cycle number from endgame
%   path_flag -                     1 if successful, 0 otherwise
%
% The format of raw_data is defined in Appendix D of "Numerically solving polynomial systems with
% Bertini", by Daniel J. Bates, Jonathan D. Haunstein, Andrew J. Sommese
% and Charles W. Wampler (SIAM 2013). There is some material at the bottom
% of the file that READ_RAW_DATA does not read.
%
% See also: solve, read_solutions, match_solutions

% Finite solutions
fname = poly_sys.fullname('raw_data');
[fid,msg] = fopen(fname,'r');
if fid < 3
    error('BertiniLab:read_raw_data:openError', msg)
end

nVariables = fscanf(fid,'%d\n',1);
fgetl(fid);

path_number = fscanf(fid,'%d\n',1);
while path_number ~= -1
    solution_info(path_number+1).path_number = path_number; %#ok<*AGROW>
    solution_info(path_number+1).max_precision = fscanf(fid,'%d\n',1);
    solution_info(path_number+1).homogeneous_coordinates = BertiniLab.read_data_point(fid,nVariables);
    solution_info(path_number+1).function_residual = fscanf(fid,'%f\n',1);
    solution_info(path_number+1).condition_number = fscanf(fid,'%f\n',1);
    solution_info(path_number+1).final_Newton_residual = fscanf(fid,'%f\n',1);
    solution_info(path_number+1).final_pathvariable = fscanf(fid,'%f\n',1);
    solution_info(path_number+1).lastdiff = fscanf(fid,'%f\n',1);    
    solution_info(path_number+1).pathvariable_of_first_precision_increase = fscanf(fid,'%f\n',1);
    solution_info(path_number+1).endgame_cycle_number = fscanf(fid,'%d\n',1);
    solution_info(path_number+1).path_flag = fscanf(fid,'%d\n',1);
    path_number = fscanf(fid,'%d\n',1);
end
    
fclose(fid);
