function poly_sys = make_input_file(poly_sys,varargin)
%MAKE_INPUT_FILE Create input file for Bertini run
%
% poly_sys = poly_sys.make_input_file_name creates an input file with the
% default name 'input' and sets poly_sys.input_file_name = 'input'.
%
% poly_sys = poly_sys.make_input_file(fname) creates an input file with name FNAME (a
% string). Saves FNAME in poly_sys.input_file_name.
%
% Example:
%   (This function is generally called by solve.)
%
% See also: solve

assert(~isempty(poly_sys.function_def),'BertiniLab:make_input_file:noFunction', ...
    'A function must be defined.')
 
% Assemble variables, declarations and assignments; save order of variable
% declarations.
[declarations,assignments] = make_declarations(poly_sys);

%----- Create file --------------
if nargin>1
    poly_sys.input_file_name = varargin{1};
else
    poly_sys.input_file_name = 'input';
end
fname = poly_sys.fullname(poly_sys.input_file_name);
fid = fopen(fname,'w');

%----- Configuration block ------
fprintf(fid,'CONFIG\n');
str = BertiniLab.format_struct(poly_sys.config);
for ii=1:size(str,1)
    fprintf(fid,'%s: %s;\n',str{ii,1},str{ii,2});
end
fprintf(fid,'END;\n\n');

%--------- Input block ----------
fprintf(fid,'INPUT\n');
% Declarations
for ii=1:length(declarations)
    poly_sys.print_declarations(fid,declarations{ii},poly_sys.(declarations{ii}),';');
end

% Assignment declarations
for ii=1:length(assignments)
    p = poly_sys.(assignments{ii});
    poly_sys.print_declarations(fid,assignments{ii},p(:,1),';');
end

% Function declarations
str = format_array(poly_sys.function_name.',', ');
fprintf(fid,'function %s;\n',str);

% Assignments
assignments = [assignments {'subfunction'}]; % Subfunctions are not declared
for ii=1:length(assignments)
    p = poly_sys.(assignments{ii});
    str = format_equations(p(:,1),p(:,2));
    fprintf(fid,'%s;\n',str{:});
end

% Functions
str = format_equations(poly_sys.function_name,poly_sys.function_def);
fprintf(fid,'%s;\n',str{:});

fprintf(fid,'END;\n');
fclose(fid);
