function varargout = evaluate(poly_sys,points,varargin)
%EVALUATE Evaluate system (and possibly Jacobian) at selected points
%
% fvals = poly_sys.evaluate(points) inputs POINTS, a structure or
% numeric/sym/polysym array of complex numbers representing one or more
% points. Finds FVALS, a vector of values of these points.
%
% [fvals,J] = poly_sys.evaluate(points) also returns J, a cell array of
% Jacobians.
%
% All outputs are POLYSYM arrays.
%
% If the job is a user homotopy,
%   [...] = poly_sys.evaluate(points,starting_point) also inputs a starting
%   point.
%
% If the job is a parameter homotopy,
%   [fvals,J,Jp] = poly_sys.evaluate(points,params) inputs PARAMS, the
%   parameter values. Returns J, the Jacobian with respect to the
%   variables, and JP, the Jacobian with respect to the parameters.
%
% [...] = poly_sys.evaluate(...,filename) also inputs an
% alternative name for the input file (default is 'input').
%
% POINTS can be a numeric, polysym or sym matrix with a column for each
% point; or it can be a structure with a field for each variable (which may
% be an array of any of the above types).
%
% EVALUATE does not store the diagnostic output from Bertini. To see this
% output displayed in the command window, set the property VIEW_PROGRESS
% to TRUE.
%
% Example:
%    polysyms x y
%    circle = x^2 + (y-1)^2 - 1;
%    parabola = y-2*x^2;
%             
%    circle_parabola_intersection = BertiniLab('function_def',[circle; parabola], ...
%       'variable_group',[x y]);
%    [fvals,J] = circle_parabola_intersection.evaluate([.75 2; 1e-15 .25].');

assert(nargin > 1,'BertiniLab:evaluate:minrhs','Not enough input arguments.')

if nargout > 1
    poly_sys.config.TrackType = -3;
else
    poly_sys.config.TrackType = -4;
end

poly_sys.starting_points = points;

% More setup for user homotopy and parameter homotopy
cf = poly_sys.config;
if isfield(cf,'UserHomotopy') && ismember(cf.UserHomotopy,[1 2]) && nargin > 2
    poly_sys.make_data_file(varargin{1},'start_time');
    varargin = varargin(2:end);
    
elseif isfield(cf,'ParameterHomotopy') && ismember(cf.ParameterHomotopy,[1 2])
    assert(nargin>2,'BertiniLab:evaluate:maxrhs', ...
        'For a parameter homotopy, the parameter values must be given.')
    if nargin > 3
        start_parameter_file_name = poly_sys.fullname(varargin{2});
    else
        start_parameter_file_name = poly_sys.fullname('start_parameters');
    end
    assert(exist(start_parameter_file_name,'file')==2, ...
        'BertiniLab:evaluate:fileNotFound', ...
        'The ab initio parameter run must be done before calling this function.')
    poly_sys.config.ParameterHomotopy=2;
    poly_sys.make_param_file(varargin{1}(:),'start_parameters');
    poly_sys.final_parameters = varargin{1};
    varargin = varargin(2:end);
    
end % Otherwise, ignore this parameter

poly_sys = poly_sys.solve(varargin{:});

varargout{1} = poly_sys.read_solutions('function');

if nargout > 1
    nVars = numel(poly_sys.order_of_variables);
    J = num2cell(poly_sys.read_solutions('Jv',nVars^2),1);
    
    % Reshape into matrices
    varargout{2} = cellfun(@(x) reshape(x,nVars,nVars).',J,'UniformOutput',false);
end

if nargout > 2
    nParams = numel(poly_sys.parameter);
    Jp = num2cell(poly_sys.read_solutions('Jp',nParams),1);
    
    % Reshape into matrices
    varargout{3} = cellfun(@(x) reshape(x,nParams,[]).',Jp,'UniformOutput',false);
end
