function varargout = Newton_iteration(poly_sys,points,varargin)
%Newton_iteration Perform a Newton iteration at selected points
%
% newPoints = poly_sys.Newton_iteration(points) inputs POINTS, a structure or
% numeric/sym/polysym array of complex numbers representing one or more
% points. Finds newPoints, a vector of values of these points.
%
% [newPoints,successFlag,NewtonResidual,conditionNumber] =
% poly_sys.Newton_iteration(points), depending on how many output arguments
% there are, also returns some of SUCCESSFLAG, a Boolean array;
% NEWTONRESIDUAL, the Newton residual for each point; and CONDITIONNUMBER,
% an approximate condition number for each point.
%
% All outputs are POLYSYM arrays except for SUCCESSFLAG, which is logical
% (1 for success, 0 for failure).
%
% If the job is a user homotopy,
%   [...] = poly_sys.Newton_iteration(points,starting_point) also inputs a
%   starting point.
%
% If the job is a parameter homotopy,
%   [...] = poly_sys.Newton_iteration(points,params) inputs PARAMS, the
%   parameter values.
%
% [...] = poly_sys.Newton_iteration(...,filename) also inputs an
% alternative name for the input file (default is 'input').
%
% POINTS can be a numeric, polysym or sym matrix with a column for each
% point; or it can be a structure with a field for each variable (which may
% be an array of any of the above types).
%
% NEWTON_ITERATION does not store the diagnostic output from Bertini, so it
% is displayed in the command window.  To see this output displayed in the
% command window, set the property VIEW_PROGRESS to TRUE.
%
% Example:
%    polysyms x y
%    circle = x^2 + (y-1)^2 - 1;
%    parabola = y-2*x^2;
%             
%    circle_parabola_intersection = BertiniLab('function_def', ...
%       [circle; parabola],'variable_group',[x y]);
%    [newPoints,successFlag,NewtonResidual] = ...
%       circle_parabola_intersection.Newton_iteration([.75 2; 1e-15 .25].');

assert(nargin > 1,'BertiniLab:Newton_iteration:minrhs','Not enough input arguments.')

if nargout > 3
    poly_sys.config.TrackType = -1;
else
    poly_sys.config.TrackType = -2;
end

poly_sys.starting_points = points;

% More setup for user homotopy and parameter homotopy
cf = poly_sys.config;
if isfield(cf,'UserHomotopy') && ismember(cf.UserHomotopy,[1 2]) && nargin > 2
    poly_sys.make_data_file(varargin{1},'start_time');
    varargin = varargin(2:end);
    
elseif isfield(cf,'ParameterHomotopy') && ismember(cf.ParameterHomotopy,[1 2])
    assert(nargin>2,'BertiniLab:evaluate:maxrhs', ...
        'For a parameter homotopy, the parameter values must be given.')
    assert(exist(poly_sys.fullname('start_parameters'),'file')==2, ...
        'BertiniLab:evaluate:fileNotFound', ...
        'The ab initio parameter run must be done before calling this function.')
    poly_sys.config.ParameterHomotopy=2;
    poly_sys.make_param_file(varargin{1}(:),'start_parameters');
    poly_sys.final_parameters = varargin{1};
    varargin = varargin(2:end);
    
end % Otherwise, ignore this parameter

poly_sys = poly_sys.solve(varargin{:});

% Return solution in same form as starting point.
if isstruct(points)
    varargout{1} = poly_sys.match_solutions('newPoints');
else
    varargout{1} = poly_sys.read_solutions('newPoints');
end

if nargout > 1
    successFlag = poly_sys.read_solutions('successFlag',1);
    varargout{2} = logical(double(successFlag)+1);
end

if nargout > 2
    varargout{3} = poly_sys.read_solutions('newtonResidual',1);
end

if nargout > 3
    varargout{4} = poly_sys.read_solutions('CN',1);
end
