classdef BertiniLab
    %BERTINILAB Interface for running Bertini
    %
    % Tools are provided for defining the problem, running Bertini and
    % examining the output. A collection of properties correspond in an
    % obvious way to definitions in the Bertini input file. All features of
    % Bertini are supported, including solving for isolated points,
    % finite-dimensional solutions and interactive dialogs with Bertini.
    % There are methods to read many of the Bertini output files and
    % assemble solutions into vectors and matrices as appropriate.
    %
    % Example:
    %   polysyms x y
    %   poly_system = BertiniLab('function_def',[x^2-1; x+y-1],'variable_group',[x y]);
    %   poly_system = poly_system.solve;
    %   disp(poly_system.solve_summary)
    %   sols = poly_system.read_solutions('finite_solutions')
    %
    %  For many more examples, see <a href="matlab:showdemo
    %  examplesFromBook">BertiniLab demos</a>.
    %
    % BertiniLab Properties:
    %   Bertini_io_folder -     Folder containing Bertini input/output files
    %   components -            Components of an irreducible decomposition
    %   config -                Configuration settings defined by Bertini
    %   constant -              Constants declared and assigned values in Bertini input
    %   definedSubfunction -    Root of subfunction filename
    %   final_parameters -      Final parameter values for a parameter homotopy
    %   function_def -          Equations to be solved
    %   function_name -         Names of functions
    %   hom_variable_group -    Homogenized variable group
    %   interactive_choices -   Responses to Bertini interactive prompts
    %   member_points -         Points to test for membership inpositive-dimensional sets
    %   output_file_names -     Names of files created during Bertini run
    %   parameter -             Parameter for user homotopy
    %   pathvariable -          Path variable for user homotopy
    %   random -                Complex random number
    %   random_real -           Real random number
    %   solve_summary -         Summary of solutions
    %   starting_points -       Initial values for a user homotopy
    %   subfunction -           Subfunctions for use in function_def expressions
    %   variable -              Variables for user-defined homotopy
    %   variable_group -        Non-homogenized variable group
    %   witness_points -        Witness points for an irreducible decomposition
    %
    %
    % BertiniLab Methods:
    %   dehomogenize -          Convert homogeneous coordinates into affine coordinates
    %   evaluate -              Evaluate system (and possibly Jacobian) at selected points
    %   irreducible_decomposition - Find the irreducible decomposition of a system
    %   match_solutions -       Assign values of solutions to the variable names
    %   mat2struct -            Convert matrix to structure
    %   membership -            Find which components some points belong to
    %   Newton_iteration -      Perform a Newton iteration at selected points
    %   read_incidence_matrix - Read information on components containing given points
    %   read_raw_data -         Read solutions with diagnostic data
    %   read_solutions  -       Read solutions from Bertini run
    %   read_witness_data -     Read information on witness points
    %   sample -                Sample an algebraic component
    %   sharpen -               Sharpen solutions of a homotopy using the sharpening module
    %   solve -                 Solve using Bertini
    %   struct2mat -            Convert structure to matrix
    %
    % See also: polysym
    
    properties
        %config - Configuration parameters used by Bertini
        %CONFIG is a structure with each assignment represented by the name
        %of a field and its value.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   poly_system.config.TRACKTYPE = 1;
        config = struct;
        %function_def - Equations to be solved
        %A POLYSYM array. It can have a single column with the equations to
        %be solved or one column with a list of equation names and one with
        %the equations.
        %
        % Example:
        %   f  = @(x) x^2-1;
        %   polysyms x
        %   poly_system = BertiniLab;
        %   poly_system.function_def = f(x);                % one column version
        %   function_name = polysyms f;
        %   poly_system.function_def = [function_name f(x)] % two column version
        function_def = polysym.empty;
        %variable - Variable declarations for user-defined homotopy
        %POLYSYM vector.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   poly_system.variable = polysym({'x','y'});
        %
        % See also: variable_group, hom_variable_group
        variable = polysym.empty;
        %variable_group - Variable declarations for non-homogeneous equations.
        %POLYSYM vector or cell array of POLYSYM vectors. If any variables
        %appear homogeneously in the polynomials, they can be declared
        %separately in hom_variable_group. Multiple variable groups can be
        %nested in cell arrays.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   poly_system.variable_group = {polysym('x','y','z')}; % Single group
        %   poly_system.variable_group = {polysym({'x','y'}) polysym('z')}; % x,y in one group and z in another.
        %   polysyms x y z    % Equivalent calls using POLYSYM arrays
        %   poly_system.variable_group = [x y z];
        %   poly_system.variable_group = {[x y], z};
        %
        %See also: hom_variable_group, variable
        variable_group = polysym.empty;
        %hom_variable_group - Variable declarations for homogeneous
        %equations.
        %POLYSYM vector or cell array of POLYSYM vectors. If any variables
        %appear inhomogeneously in the polynomials, they can be declared
        %separately in hom_variable_group. Multiple variable groups can be
        %nested in cell arrays.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   poly_system.hom_variable_group = {polysym('x','y','z')}; % Single group
        %   poly_system.hom_variable_group = {polysym({'x','y'}) polysym('z')}; % x,y in one group and z in another.
        %   polysyms x y z    % Equivalent calls using POLYSYM arrays
        %   poly_system.hom_variable_group = [x y z];
        %   poly_system.hom_variable_group = {[x y], z};
        %
        %See also: hom_variable_group, variable_group
        hom_variable_group = polysym.empty;
        %pathvariable - Path variable for user homotopy
        %POLYSYM scalar.
        %See also: parameter
        pathvariable = polysym.empty;
        %random - Complex random number
        %POLYSYM scalar.
        %See also: random_real
        random = polysym.empty;
        %random_real - Real random number
        %POLYSYM scalar.
        %See also: random_real
        random_real = polysym.empty;
        %definedSubfunction - Root of subfunction filename
        %POLYSYM scalar, representing a string minus the file extent
        %'.func'.
        %
        % Example:
        %   definedSubfunction = Z; % File 'Z.func' must exist
        definedSubfunction = polysym.empty;
        %parameter - Parameter for user homotopy or parameter homotopy
        %POLYSYM array with a column for the names of the parameters and
        %another for their values.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   t = polysym('t');
        %   poly_system.pathvariable = t;
        %   poly_system.parameter = [polysym('p',[1 2]).'; t^2 t^3];
        %
        parameter = polysym.empty(0,2);
        %constant - Constants that are declared and assigned values in the Bertini input file
        %POLYSYM array with one column for the names and one column for
        %their values.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   g = polysym('g',[2 1]);
        %   gvals = [1.25; 0.75-1.13*1i];
        %   poly_system.constant = [g gvals];
        constant = polysym.empty(0,2);
        %subfunction - Subfunctions for use in function_def expressions
        %POLYSYM array with one column for the names and one column for
        %their values.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   x = polysym('x');
        %   V = x^2 + 1.0;
        %   f1 = 2*v^2 + 4.0;
        %   poly_system.subfunction = [polysym({'x';'V';'f1'}) [x; V; f1]];
        subfunction = polysym.empty(0,2);
        %starting_points - Initial values for a user homotopy
        %POLYSYM or compatible array with a row of numbers for each (scalar) variable
        %
        % Example:
        %   poly_system = BertiniLab;
        %   poly_system.starting_points = [1.1 2.4];
        starting_points = polysym.empty;
        %final_parameters - Final parameter values for a parameter homotopy
        %POLYSYM vector containing the numerical values.
        %
        % Example: (for two points (.75, 2) and (1e-15, .25))
        %   poly_system = BertiniLab;
        %   polysyms x y
        %   poly_system.final_parameters = [.75 1e-15; 2 .25];
        final_parameters = polysym.empty;
        %member_points - Points to test for membership in
        %positive-dimensional sets.
        %POLYSYM vector containing the numerical values.
        %
        % Example:
        %   poly_system = BertiniLab;
        %   polysyms x y
        %   poly_system.member_points = [1.1 2.4 0.8];
        member_points = polysym.empty;
        %interactive_choices - Responses to Bertini interactive prompts
        %Cell array of strings, each string being the response to one
        %question.
        interactive_choices = {};
        %Bertini_io_folder - folder containing Bertini input/output files
        %String containing the name of a folder. The default is an empty
        %string, in which case the I/O occurs in the current working
        %folder.
        %
        %A folder name may have the relative or absolute path. If the
        %folder does not exist, it is created the next time the property
        %io_folder is accessed. If a relative path name is provided, a new
        %I/O folder will be created each time the user changes the working
        %folder.
        Bertini_io_folder = '';
        %view_progress - watch progress as Bertini executes
        %Logical variable. If true, output of Bertini is echoed to the
        %screen as well as being saved to solve_summary. This output
        %includes notices of the status of the computation - useful if the
        %computation takes a long time.
        view_progress = false;
        %components - components of irreducible decomposition
        %Structure containing information on each component. The fields
        %are:
        %   codim -                         integer (counts up from 0)
        %   dim -                           integer (derived from codim)
        %   endpoint -                      polysym array representing a complex solution
        %   last_approx -                   polysym array representing a complex solution
        %   endpoint_precision -            precision used for the endpoint
        %   last_approx_precision -         precision used for last approximation
        %   condition_number -              double precision
        %   corank -                        integer
        %   smallest_singular -             double precision
        %   largest_singular -              double precision
        %   type -                          integer
        %   multiplicity -                  integer
        %   component_number -              integer
        %   deflations_needed -             integer
        components = struct.empty;
    end
    
    properties (SetAccess=protected)
        %solve_summary - Summary of run by Bertini
        %A character array containing the output that Bertini would
        %normally send to the standard output. Contains tables of solutions
        %by classification.
        solve_summary = '';
    end
    
    % Diagnostics and information on the variables
    properties (Hidden)       
        %has_failed_paths, has_path_crossings - Bertini diagnostics
        %Logical variables indicating a problem with the solution.
        has_failed_paths = false;
        has_path_crossings = false;
        
        %input_file_name - name of Bertini input file
        input_file_name = 'input';
    end
    
    % Edit the properties below to give the location of Bertini.
    properties(Constant, Hidden)
        program_folder = '/usr/local/bin';                % Mac or Unix example
        % program_folder = 'C:\cygwin64\usr\local\bin';   % Windows example
        program_name = 'bertini';
    end
    
    properties(Dependent, SetAccess=private, Hidden)
        %bertini String for calling Bertini
        %String, giving name by which Bertini can be called in the current
        %folder (derived from program_name and program_folder).
        bertini
        %io_folder Name of folder in which to put Bertini files
        io_folder
        %output_file_names Names of files created during Bertini run
        %Cell array of strings
        output_file_names
        %echo_flag Flag for system command
        %Either '-echo' or empty string. If the former, output from Bertini
        %is echoed to the screen.
        echo_flag
    end
    
    properties(Dependent)
        %function_name Names of functions in Bertini input file
        %POLYSYM vector, usually created automatically.
        function_name
        %order_of_variables - Order in which variables were declared to Bertini
        %POLYSYM array containing variables in the order they appear in the
        %Bertini input file.
        order_of_variables
        %witness_points - Witness points for irreducible decomposition
        %Structure having the same fields as for isolated point solutions,
        %with one element per component.
        witness_points
    end
    
    properties(Access=private)
        function_name_userdefined = polysym.empty;
    end
    
    methods
        function poly_sys = BertiniLab(varargin)
            %BERTINILAB Constructor.
            % poly_system = BertiniLab creates an empty interface object.
            %
            % poly_system = BertiniLab(varargin) inputs name/value pairs for
            % BertiniLab properties, where the name is a string and the
            % allowed classes for the variable depend on the property (see
            % the documentation for each property).
            %
            % Example:
            %   poly_system = BertiniLab('function_def',{'x^2-1'; 'x+y-1'}, ...
            %                   'variable_group',{'x','y'});
            %
            % See also: BertiniLab
            
            if nargin > 0
                p = varargin(1:2:end);
                v = varargin(2:2:end);
                
                assert(numel(p)==numel(v),'BertiniLab:Constructor:notPaired', ...
                    'Arguments should be in name/value pairs.')
                
                for ii=1:numel(p)
                    poly_sys.(p{ii}) = v{ii};
                end
            end
        end
        
        %----- Set/get methods ----------------
        function poly_sys = set.config(poly_sys,value)
            validateattributes(value,{'struct'},{'nonempty'},'set.config', ...
                'value',2)
            poly_sys.config=value;
        end
        function poly_sys = set.function_def(poly_sys,value)
            value = polysym(value);
            validateattributes(value,{'polysym'},{'vector'},'BertiniLab:set:function_name', ...
                'value',2)
            poly_sys.function_def = value(:);
        end
        function poly_sys = set.function_name(poly_sys,value)
            value = polysym(value);
            validateattributes(value,{'polysym'},{'vector'},'BertiniLab:set:function_name', ...
                'value',2)
            poly_sys.function_name_userdefined = value(:);
        end
        
        function poly_sys = set.variable(poly_sys,value)
            assert(~iscell(value)||iscellstr(value),'BertiniLab:set:variable:invalidType', ...
                'Variable cannot be a cell array of polysym or sym objects.')
            poly_sys.variable=polysym(value);
        end
        function poly_sys = set.variable_group(poly_sys,value)
            if iscell(value) && ~iscellstr(value)
                v = cell(size(value));
                for ii=1:numel(value)
                    v{ii} = polysym(value{ii});
                end
            else
                v = polysym(value);
            end
            poly_sys.variable_group=v;
        end
        function poly_sys = set.hom_variable_group(poly_sys,value)
            if iscell(value) && ~iscellstr(value)
                v = cell(size(value));
                for ii=1:numel(value)
                    v{ii} = polysym(value{ii});
                end
            else
                v = polysym(value);
            end
            
            poly_sys.hom_variable_group=v;
        end
        function poly_sys = set.pathvariable(poly_sys,value)
            v = polysym(value);
            validateattributes(v,{'polysym'},{'scalar'},'set:pathvariable', ...
                'value',2)
            poly_sys.pathvariable=v;
        end
        function poly_sys = set.random_real(poly_sys,value)
            v = polysym(value);
            validateattributes(v,{'polysym'},{'vector'},'set:random_real', ...
                'value',2)
            poly_sys.random_real=v;
        end
        function poly_sys = set.random(poly_sys,value)
            v = polysym(value);
            validateattributes(v,{'polysym'},{'vector'},'set:random', ...
                'value',2)
            poly_sys.random=v;
        end
        function poly_sys = set.definedSubfunction(poly_sys,value)
            v = polysym(value);
            validateattributes(v,{'polysym'},{'scalar'},'set:definedSubfunction', ...
                'value',2)
            s = v.Value;
            assert(exist(poly_sys.fullname([s,'.func']),'file')==2, ...
                'BertiniLab:set:definedSubfunction:nonexistentFile', ...
                ['The subfunction file ',s,'.func does not exist.'])
            poly_sys.definedSubfunction=v;
        end
        function poly_sys = set.parameter(poly_sys,value)
            poly_sys.parameter = polysym(value);
        end
        function poly_sys = set.constant(poly_sys,value)
            if isempty(value)
                poly_sys.constant = polysym.empty(0,2);
            else
                v = polysym(value);
                validateattributes(v,{'polysym'},{'ncols',2},'set:constant', ...
                    'value',2)
                poly_sys.constant = v;
            end
        end
        function poly_sys = set.starting_points(poly_sys,value)
            poly_sys.starting_points = poly_sys.struct2mat(value);
        end
        function poly_sys = set.final_parameters(poly_sys,value)
            poly_sys.final_parameters = polysym(value);
        end
        function poly_sys = set.member_points(poly_sys,value)
            poly_sys.member_points = poly_sys.struct2mat(value);
        end
        function poly_sys = set.Bertini_io_folder(poly_sys,fname)
            validateattributes(fname,{'char'},{},'set:Bertini_io_folder', ...
                'fname',2)
            
            poly_sys.Bertini_io_folder = fname;
        end
        function fnames = get.function_name(poly_sys)
            if isempty(poly_sys.function_name_userdefined)
                fnames = polysym('eqn',size(poly_sys.function_def));
            else
                fnames = poly_sys.function_name_userdefined;
            end
        end
        function iof = get.io_folder(poly_sys)
            if isempty(poly_sys.Bertini_io_folder)
                iof = pwd;
            else
                iof = poly_sys.Bertini_io_folder;
                
                % Make folder, if necessary
                if ~exist(iof,'dir')
                    mkdir(iof)
                end
            end
        end
        function names = get.output_file_names(poly_sys)
            results = poly_sys.solve_summary;
            matches = regexp(results,'\w+:','match');
            istart = find(strcmp(matches,'you:'));
            if ~isempty(istart)
                matches = matches(istart+1:end);
                
                itracked = find(strcmp(matches,'Tracked:'));
                if ~isempty(itracked)
                    matches = matches(1:itracked-1);
                end
            else
                istart = strfind(results,'Writing');
                results = results(istart:end);
                matches = regexp(results,'(?<='')\w*','match');
            end
            names = regexprep(matches,':',''); % Remove colon from name
        end
        function str = get.bertini(poly_sys)
            poly_sys.addpath_system(poly_sys.program_folder);
            BertiniLab.bertiniVersion(poly_sys.program_name);
            str = poly_sys.program_name;
        end
        function str = get.echo_flag(poly_sys)
            if poly_sys.view_progress
                str = '-echo';
            else
                str = '';
            end
        end
        function vars = get.order_of_variables(poly_sys)
            vars = polysym.empty;
            declarationTypes = {'variable_group','hom_variable_group','variable'};
            for ty = declarationTypes
                v = poly_sys.(ty{:});
                if iscell(v)
                    for jj=1:length(v)
                        vars = [vars; v{jj}(:)]; %#ok<AGROW>
                    end
                else
                    vars = [vars; v(:)]; %#ok<AGROW>
                end
            end
        end
        function points = get.witness_points(poly_sys)
            comp = poly_sys.components;
            for ii=length(comp):-1:1
                points(ii) = poly_sys.mat2struct(comp(ii).endpoint);
            end
        end
        % Main solver options
        poly_sys = solve(poly_sys,varargin)
        varargout = evaluate(poly_sys,points,varargin)
        varargout = Newton_iteration(poly_sys,points,varargin)
        poly_sys = irreducible_decomposition(poly_sys,varargin)
        
        % I/O methods
        sstruct = match_solutions(poly_sys,filename,varargin)
        incidence_info = read_incidence_matrix(poly_sys,fname)
        solution_info = read_raw_data(poly_sys);
        data = read_solutions(poly_sys,fname,ndims)
        [solution_info,components] = read_witness_data(poly_sys)
        
        % Other methods
        data = dehomogenize(poly_sys,data)
        sstruct = mat2struct(poly_sys,M)
        M = struct2mat(poly_sys,sstruct)
        sample_points = sample(poly_sys,component,npoints,fname)
    end
    methods (Access=protected)
        [variables,declarations,assignments] = make_declarations(poly_sys)
    end
    methods (Hidden)
        interactive_file_name = make_interactive_choices(poly_sys,choices)
        fname = make_data_file(poly_sys,data,fname)
        fname = make_param_file(poly_sys,params,fname)
        poly_sys = make_input_file(poly_sys,varargin)
        fname = fullname(poly_sys,fname)
    end
    methods(Static, Hidden)
        bertiniVersion(filename)
        systemPath = addpath_system(pathname)
        out = format_struct(in)
        print_declarations(fid,declaration_type,vars,terminator)
        y = read_data_point(fid,nComponents)
    end
    
end
