
/* 
        Provides an interface to the GP sparse direct LU solver

	Sivan Toledo, 1996
*/

#include <src/mat/impls/aij/seq/aij.h>
/*#include <plog.h>*/
#include <math.h>
#include <assert.h>

#define DESC_TYPE_CSC 10001

#define CSC_DESC_TYPE 0
#define CSC_DESC_M    1
#define CSC_DESC_N    2
#define CSC_DESC_NNZ  3
#define CSC_DESC_BASE 4
#define CSC_DESC_COLPTR 5
#define CSC_DESC_ROWIND(desc) (5+((desc)[2])+1)

#if defined(HAVE_GP) && !defined(__cplusplus)

typedef struct {
  void* lu;
  int*  perm_c;
  int   perm_c_length;
} gp_spptr_t;

/*** error handling ***/

static void MatStatisticReporter_SeqAIJ_GP(double ctxt[], char* name, double* v)
{
  if (! strcmp(name,"FLOPS") )   ctxt[0] = *v;
  if (! strcmp(name,"NONZEROS") ) ctxt[1] = *v;
}

extern int MatDestroy_SeqAIJ(PetscObject);

static int MatDestroy_SeqAIJ_GP(PetscObject obj)
{
  Mat                A  = (Mat) obj;
  Mat_SeqAIJ         *a = (Mat_SeqAIJ*) A->data;

  gp_spptr_t*        gp_spptr = (gp_spptr_t*) a->spptr;

  printf("gp: mat destroy\n");

  PetscFree( gp_spptr->perm_c );

  gstff_gp_c(gp_spptr->lu);

  PetscFree( gp_spptr );

  MatDestroy_SeqAIJ( obj );

  return 0;
}

static int MatSolve_SeqAIJ_GP(Mat A,Vec b,Vec x)
{
  Mat_SeqAIJ         *a = (Mat_SeqAIJ*) A->data;
  Scalar             *xarray;
  int                ierr,m, zero = 0;
  int i;
  double flops;
  double time;
  PLogDouble t1,t2;

  gp_spptr_t*        gp_spptr = (gp_spptr_t*) a->spptr;

  void* gp;

  /*printf("gp: mat solve\n");*/

  if (A->factor != FACTOR_LU) 
    SETERRQ(PETSC_ERR_ARG_WRONG,
	    1,
	    "MatSolve_SeqAIJ_GP:Not for unfactored matrix");

  if (gp_spptr == NULL) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatSolve_SeqAIJ_GP:Factor not found (1)");

  if (gp_spptr->lu == NULL) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatSolve_SeqAIJ_GP:Factor not found (2)");

  ierr = VecGetSize(b,&m); CHKERRQ(ierr);
  ierr = VecCopy(b,x); CHKERRQ(ierr);
  ierr = VecGetArray(x,&xarray); CHKERRQ(ierr);

  ierr = PetscGetTime(&t1);

  gp_create_c(&gp,&ierr); CHKERRQ(ierr);
  gp_set_statistics_reporter_c(gp,&MatStatisticReporter_SeqAIJ_GP,&flops);

  dgstrs_gp_c(gp,
	      "Transpose",
	      m,1,
	      gp_spptr->lu,1,1,
	      xarray,1,1,-1,
	      &ierr);

  gp_destroy_c(gp);

  PetscGetTime(&t2);
  time = t2 - t1;

  PLogFlops((int) flops);

  if (ierr < 0) {
    fprintf(stderr,"error = %d\n",ierr);
    SETERRQ(PETSC_ERR_LIB,1,"MatSolve_SeqAIJ_GP:input error in GP:dgstrs"); 
  }

  ierr = VecRestoreArray(x,&xarray); CHKERRQ(ierr);

  return 0;
}

static int MatLUFactorNumeric_SeqAIJ_GP(Mat A,Mat *F)
{
  Mat_SeqAIJ         *f       = (Mat_SeqAIJ*) (*F)->data;
  Mat_SeqAIJ         *a       = (Mat_SeqAIJ*) (A)->data;

  gp_spptr_t*        gp_spptr = (gp_spptr_t*) f->spptr;

  void* gp;

  int                i,j,ptr,desc_ptr,ierr,flg;
  double             dt;

  double flops[2];
  double time;
  PLogDouble t1,t2;

  int*    a_desc;
  double* a_nz;

  /*
  extern double nnz;
  extern double fact_time;
  */

  /*printf("gp: num fact\n");*/

  ierr = PetscGetTime(&t1);

  if (gp_spptr == NULL) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatFactorNumeric_SeqAIJ_GP:Symbolic factor not found");

  if (gp_spptr->perm_c == NULL) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatFactorNumeric_SeqAIJ_GP:Column ordering not found");

  a_desc = PetscMalloc( (6 + (a->n) + 1 + a->nz) 
                        * sizeof( int ) ); CHKPTRQ( a_desc );

  a_nz   = PetscMalloc( (a->nz) 
                        * sizeof( double ) ); CHKPTRQ( a_desc );

  a_desc[CSC_DESC_TYPE] = DESC_TYPE_CSC;
  a_desc[CSC_DESC_M]    = a->n;
  a_desc[CSC_DESC_N]    = a->m;
  a_desc[CSC_DESC_NNZ]  = a->nz;
  a_desc[CSC_DESC_BASE] = 0;

  a_desc[CSC_DESC_COLPTR] = 0;
  for (j=1; j<=(a->m); j++)
    a_desc[CSC_DESC_COLPTR+j] = a_desc[CSC_DESC_COLPTR+j-1] + (a->ilen)[j-1];

  desc_ptr = CSC_DESC_ROWIND(a_desc);
  ptr = 0;
  for (j=0; j<(a->m); j++) {

    assert (desc_ptr-(CSC_DESC_ROWIND(a_desc)) ==  a_desc[CSC_DESC_COLPTR+j]);
    assert (ptr < a->nz);

    for (i=0; i<((a->ilen)[j]); i++) {
      a_desc[desc_ptr++] = (a->j)[ (a->i)[j] + i];
      a_nz[ptr++]        = (a->a)[ (a->i)[j] + i];
    }
  }
  
  gp_create_c(&gp,&ierr); CHKERRQ(ierr);
  gp_set_pivot_policy_c(gp,1);
  gp_set_pivot_threshold_c(gp,A->lupivotthreshold);
  gp_set_statistics_reporter_c(gp,&MatStatisticReporter_SeqAIJ_GP,flops);
  gp_set_col_perm_c(gp,gp_spptr->perm_c,gp_spptr->perm_c_length,0,ierr);
  if (ierr) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatFactorNumeric_SeqAIJ_GP:Could not set column ordering");

  ierr = OptionsGetDouble(PETSC_NULL,"-pc_ilu_use_drop_tolerance",
			  &dt,&flg); 
  CHKERRQ(ierr);
  if (flg==0) dt=0.0; /* do not drop */
  gp_set_drop_threshold_c(gp,dt);

  ierr = OptionsGetDouble(PETSC_NULL,"-pc_ilu_col_fill_ratio",
			  &dt,&flg); 
  CHKERRQ(ierr); 
  if (flg==0) dt=-1.0; /* do not limit column fill ratio */
  gp_set_col_fill_ratio_c(gp,dt);

  dgstrf_gp_c(gp,
	      a->m,
	      a->n,
	      a_nz,
	      a_desc,
	      &(gp_spptr->lu),
	      &ierr);
  PetscGetTime(&t2);
  time = t2 - t1;
  /*fact_time = time;*/

  gp_destroy_c(gp);

  PetscFree( a_desc );
  PetscFree( a_nz );

  PLogFlops((int) flops[0]);

  /*nnz = flops[1];*/

  if (ierr == -999)
    SETERRQ(PETSC_ERR_MEM,
	    1,"MatLUFactorNumeric_SeqAIJ_GP:out of memory in GP"); 

  if (ierr < 0)
    SETERRQ(PETSC_ERR_LIB,1,"MatLUFactorNumeric_SeqAIJ_GP:input error in GP"); 

  if (ierr > 0)
    SETERRQ(PETSC_ERR_MAT_LU_ZRPVT,1,
	    "MatLUFactorNumeric_SeqAIJ_GP:zero pivot encountered"); 
  return 0;
}

static int MatLUFactorSymbolic_SeqAIJ_GP(Mat A,IS r,IS c,double f,Mat *F)
{
  Mat             B;
  Mat_SeqAIJ      *a = (Mat_SeqAIJ*) A->data, *b;
  int             ierr, *ridx, *cidx,i, len;
  double time;

  gp_spptr_t*     gp_spptr;

  int* is_r;

  /*printf("gp: sym fact\n");*/

  if (a->m != a->n) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatLUFactorSymbolic_SeqAIJ_GP:matrix must be square"); 
  if (a->indexshift != 0) 
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE,
	    1,"MatLUFactorSymbolic_SeqAIJ_GP:matrix must have 0-based indices"); 

  ierr = MatCreateSeqAIJ(A->comm,a->m,a->n,0,PETSC_NULL,F); CHKERRQ(ierr);

  gp_spptr = PetscNew(gp_spptr_t); CHKPTRQ(gp_spptr);
  gp_spptr->perm_c_length = (a->m); 
  gp_spptr->perm_c = PetscMalloc((a->m)*sizeof(int));CHKPTRQ(gp_spptr->perm_c);

  B             = *F;
  B->ops->solve  = MatSolve_SeqAIJ_GP;
  B->bops->destroy = MatDestroy_SeqAIJ_GP;
  B->factor     = FACTOR_LU;
  b             = (Mat_SeqAIJ*) B->data;
  b->spptr      = (void*) gp_spptr;

  ierr = ISGetIndices(r,&is_r); CHKERRQ(ierr);

  for (i=0; i<(a->m); i++) (gp_spptr->perm_c)[is_r[i]] = i;
  for (i=0; i<(a->m); i++) (gp_spptr->perm_c)[i] = is_r[i];

  ierr = ISRestoreIndices(r,&is_r); CHKERRQ(ierr);

  A->ops->lufactornumeric  = MatLUFactorNumeric_SeqAIJ_GP;
  B->ops->lufactornumeric  = MatLUFactorNumeric_SeqAIJ_GP;

  return 0;
}

int MatUseGP_SeqAIJ(Mat A)
{
  PetscValidHeaderSpecific(A,MAT_COOKIE);  
  if (A->type != MATSEQAIJ) return 0;

  A->ops->lufactorsymbolic = MatLUFactorSymbolic_SeqAIJ_GP;
  A->ops->lufactornumeric  = MatLUFactorNumeric_SeqAIJ_GP;

  return 0;
}

#else

int MatUseGP_SeqAIJ(Mat A)
{
  return 0;
}

#endif




