C  -*- Mode: FORTRAN; Package: JRG -*-
c
c  This file contains one routine:
c               main
c
c***********************************************************************
c     Sparse Pivoting in Time Proportional to Arithmetic Operations
c
c     This is a sample driver for LUFACT and LUSOLV, to illustrate how
c     they are called.  It reads a matrix from a file in a format
c     described below into the arrays A, AROW, and ACOLST, then factors
c     this matrix with LUFACT, overwriting the original values in A and
c     AROW.  For a description of the matrix data structure, see LUFACT.
c
c     If the matrix is square and no error occurs in factorization, the
c     vector composed of the sums of the rows of the original matrix is
c     solved for this factorization using LUSOLV, with which solution
c     the relative error of the factorization is then computed.
c
c     If the problem is rectangular, the matrices PtL and U resulting
c     from the factorization are written out to a file.
c
c     The variables PIVOT and THRESH may be set to do partial or
c     threshold pivoting.
c
c     Modified: 12 Nov 1987  John Gilbert: Use "second" for timing
c***********************************************************************
c
      program   main
c
c***********************************************************************
c     Storage required by this routine (with overwriting):
c
c     real*8    a(lastlu), dense(n)
c     integer   arow(lastlu), perm(n), iw(3*n partial / 4*n threshold)
c     integer   acolst(n+1), lcolst(n), ucolst(n+1)
c
      integer   maxn, maxlu, iwdim, maxnp1
      parameter (maxn=12000, maxlu=3000000)
      parameter (iwdim=4*maxn, maxnp1=maxn+1)
c
c***********************************************************************
c
      real*8    a(maxlu), thresh, x(maxn), rw(maxn)
      real*8    tstart, tfact, tsolv, ttotl, norm, tnorm, mflops
      real*8	opcnt
      integer   pivot, nrow, ncol, lastlu, error
      integer   arow(maxlu), acolst(maxnp1)
      integer   lcolst(maxn), ucolst(maxnp1)
      integer   perm(maxn), iw(iwdim)
      integer   lasta, nprobs, i, k, nnz, n, nzptr
      integer   nreals, nints, nrpi, n2rpi
      integer   title(60)
c
c
      real      second
      real*8    flops
c
c***********************************************************************
c
c
      pivot = 1
      thresh = .1
      nprobs = 1
c
c
c         --------------------
c         Read in the matrix.
c         --------------------
c
c	  up to 60 characters           title
c         integer nrow                  number of rows
c         integer ncol                  number of cols
c         (for each column:)
c               integer nnz             number of nzs in col
c                       integer index   row index of nonzero
c                       real value      value of nonzero
c
          lasta = 1
          read (5, 901, end=900) title
901       format (60a1)
          write (6, 902) title
902       format (60a1)
          read (5,*) nrow, ncol
          if (nrow.gt.maxn .or. ncol.gt.maxn) goto 851
          do 210 i = 1, ncol
              read (5,*) nnz
              acolst(i) = lasta
              if (nnz + lasta - 1 .gt. maxlu) goto 852
              do 200 k = 1, nnz
                  read (5,*) arow(lasta), a(lasta)
                  lasta = lasta + 1
200           continue
210       continue
          acolst(ncol+1) = lasta
          lasta = lasta - 1
c
c         --------------------------------------
c         Put the row sums of A in the vector X.
c         --------------------------------------
c
          call rfill (x, nrow, 0.0)
          do 310 k = 1, ncol
              do 300 nzptr = acolst(k), acolst(k+1) - 1
                  i = arow(nzptr)
                  x(i) = x(i) + a(nzptr)
300           continue
310       continue
c
c         ---------------------------------------------------
c         Perform numerical factorization (with overwriting).
c         ---------------------------------------------------
          tstart = second ()
          call lufact (pivot, thresh, nrow, ncol, a, arow, acolst,
     1                 maxlu, lastlu, a, arow, lcolst, ucolst, perm,
     1                 error, rw, iw)
          tfact = second () - tstart
c
c         --------------------------------------------------
c         Set RELERR and TSOLV to appropriate values in case
c         problem is rectangular or there is a zero pivot.
c         --------------------------------------------------
c
          norm = 0.0
          tsolv = 0.0
          if  ( error .ne. 0 .or. nrow .ne. ncol)  go to 500
c
c         ------------------------------------------------------------
c         Find solution and compute infinity norm of X - (1,1,...,1)'.
c         ------------------------------------------------------------
c
          tstart = second ()
          call lusolv (ncol, a, arow, lcolst, ucolst, perm, x, rw)
          tsolv = second () - tstart
c
          do 400 k = 1, nrow
              tnorm = abs(x(k) - 1.0)
              if (tnorm .gt. norm) norm = tnorm
400       continue
c
c         -----------------
c         Print statistics.
c         -----------------
500       continue
          opcnt = flops (nrow, ncol, arow, lcolst, ucolst, iw)
          mflops = 1.e-6 * opcnt / tfact
          ttotl = tfact + tsolv
          nreals = lastlu + nrow
          nints = lastlu + 7*nrow + 2
          if (pivot .eq. 2)
     1        nints = nints + nrow
          nrpi = nreals + nints
          n2rpi = 2*nreals + nints
          write (6, 599) nprobs, error, nrow, ncol, lasta,
     1                  lastlu, opcnt, tfact, tsolv, ttotl, 
     1                  mflops, norm,
     1                  nreals, nints, nrpi, n2rpi
599       format ( / 5x, 'Problem       ', i10
     1             / 5x, 'Errflag       ', i10
     1             / 5x, 'NROW  ', i10
     1             / 5x, 'NCOL  ', i10
     1             / 5x, 'M             ', i10
     1             / 5x, 'M*            ', i10
     1             / 5x, 'Ops(factor)   ', f11.0
     1             / 5x, 'T(factor)     ', f14.3
     1             / 5x, 'T(solve)      ', f14.3
     1             / 5x, 'T(total)      ', f14.3
     1             / 5x, 'Mflops(factor)', f14.3
     1             / 5x, 'Rel. error    ', 1pe18.3
     1             / 5x, '#Reals        ', i10
     1             / 5x, '#Integers     ', i10
     1             / 5x, '#R + #I       ', i10
     1             / 5x, '2#R + #I      ', i10 )
c
c         --------------------------------
c         End of loop for square problems.
c         --------------------------------
c
          if (nrow .eq. ncol) go to 900
c
c         -----------------------------------
c         Invert the permutation for results.
c         -----------------------------------
c
          do 600 i = 1, nrow
              iw(perm(i)) = i
600       continue
c
c         --------------------------------------------
c         Output the results (problem is rectangular).
c         --------------------------------------------
c
          n = min(nrow, ncol)
          write (10,*) nrow, n
          do 710 i = 1, n - 1
              write (10,*) ucolst(i+1) - lcolst(i) + 1
              write (10,*) iw(i), 1.0
              do 700 k = lcolst(i), ucolst(i+1) - 1
                  write (10,*) iw(arow(k)), a(k)
700           continue
710       continue
          write (10,*) 1
          write (10,*) iw(n), 1.0
c
          write (10,*) n, ncol
          do 810 i = 1, ncol
              write (10,*) lcolst(i) - ucolst(i)
              do 800 k = ucolst(i), lcolst(i) - 1
                  write (10,*) arow(k), a(k)
800           continue
810       continue
c
          goto 900
c
851   write (*,*) 'lufact:  only room for ', maxn, ' rows or columns.'
      goto 900
c
852   write (*,*) 'lufact:  only room for ', maxlu, ' nonzeros.'
c
900   continue
      stop
      end
