#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex1.c,v 1.70 1998/04/28 03:49:10 curfman Exp $";
#endif

/* Program usage:  mpirun ex1 [-help] [all PETSc options] */

static char help[] = "Solves a tridiagonal linear system with SLES.\n\n";

/*T
   Concepts: SLES^Solving a system of linear equations (basic uniprocessor example);
   Routines: SLESCreate(); SLESSetOperators(); SLESSetFromOptions();
   Routines: SLESSolve(); SLESView(); SLESGetKSP(); SLESGetPC();
   Routines: KSPSetTolerances(); PCSetType();
   Processors: 1
T*/

/* 
  Include "sles.h" so that we can use SLES solvers.  Note that this file
  automatically includes:
     petsc.h  - base PETSc routines   vec.h - vectors
     sys.h    - system routines       mat.h - matrices
     is.h     - index sets            ksp.h - Krylov subspace methods
     viewer.h - viewers               pc.h  - preconditioners
*/
#include "sles.h"

char* termination_reason;

int main(int argc,char **args)
{
  Vec     x, b, u;      /* approx solution, RHS, exact solution */
  Mat     A;            /* linear system matrix */
  SLES    sles;         /* linear solver context */
  PC      pc;           /* preconditioner context */
  KSP     ksp;          /* Krylov subspace method context */
  double  bnorm,rnorm;
  int     ierr, ierrsolve, i, n = 10, col[3], its, flg, size;
  Scalar  neg_one = -1.0, one = 1.0, value[3];

  PetscInitialize(&argc,&args,(char *)0,help);
  MPI_Comm_size(PETSC_COMM_WORLD,&size);
  if (size != 1) SETERRA(1,0,"This is a uniprocessor example only!");
  ierr = OptionsGetInt(PETSC_NULL,"-n",&n,&flg); CHKERRA(ierr);

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
         Compute the matrix and right-hand-side vector that define
         the linear system, Ax = b.
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

  /* 
     Create matrix.  When using MatCreate(), the matrix format can
     be specified at runtime.
  */

  n = read_ijv("lhr01.ijv",&A);

  /* 
     Create vectors.  Note that we form 1 vector from scratch and
     then duplicate as needed.
  */
  ierr = VecCreate(PETSC_COMM_WORLD,PETSC_DECIDE,n,&x); CHKERRA(ierr);
  ierr = VecDuplicate(x,&b); CHKERRA(ierr);
  ierr = VecDuplicate(x,&u); CHKERRA(ierr);

  /* 
     Set exact solution; then compute right-hand-side vector.
  */
  ierr = VecSet(&one,u); CHKERRA(ierr);
  ierr = MatMult(A,u,b); CHKERRA(ierr);

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
                Create the linear solver and set various options
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* 
     Create linear solver context
  */
  ierr = SLESCreate(PETSC_COMM_WORLD,&sles); CHKERRA(ierr);

  /* 
     Set operators. Here the matrix that defines the linear system
     also serves as the preconditioning matrix.
  */
  ierr = SLESSetOperators(sles,A,A,DIFFERENT_NONZERO_PATTERN); CHKERRA(ierr);

  /* 
     Set linear solver defaults for this problem (optional).
     - By extracting the KSP and PC contexts from the SLES context,
       we can then directly call any KSP and PC routines to set
       various options.
     - The following four statements are optional; all of these
       parameters could alternatively be specified at runtime via
       SLESSetFromOptions();
  */
  ierr = SLESGetKSP(sles,&ksp); CHKERRA(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
  ierr = PCSetType(pc,PCLU); CHKERRA(ierr);

  ierr = MatUseGP_SeqAIJ(A);

  /*
  ierr = OptionsSetValue("-mat_lu_pivotthreshold","1.0");
  ierr = OptionsSetValue("-pc_ilu_col_fill_ratio","-1.0"); 
  ierr = OptionsSetValue("-pc_ilu_use_drop_tolerance","0.0"); 
  ierr = OptionsSetValue("-pc_ilu_use_drop_tolerance","1.0"); 
  */

  ierr = KSPSetTolerances(ksp,
			  1e-10, /* rel residual tol   */
			  1e-100, /* abs residual tol   */
			  1e5,   /* divergence tol     */
			  250    /* maxits             */
			  ); 
  CHKERRA(ierr);

  /* 
    Set runtime options, e.g.,
        -ksp_type <type> -pc_type <type> -ksp_monitor -ksp_rtol <rtol>
    These options will override those specified above as long as
    SLESSetFromOptions() is called _after_ any other customization
    routines.
  */
  ierr = SLESSetFromOptions(sles); CHKERRA(ierr);
 
  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
                      Solve the linear system
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* 
     Solve linear system
  */
  ierrsolve = SLESSolve(sles,b,x,&its); CHKERRA(ierr); 

  /* 
     View solver info; we could instead use the option -sles_view to
     print this info to the screen at the conclusion of SLESSolve().
  */
  /*ierr = SLESView(sles,VIEWER_STDOUT_WORLD); CHKERRA(ierr);*/

  /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
                      Check solution and clean up
     - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
  /* 
     Check the error
  */

  /*
  ierr = VecAXPY(&neg_one,u,x); CHKERRA(ierr);
  ierr  = VecNorm(x,NORM_2,&norm); CHKERRA(ierr);
  if (norm > 1.e-12) 
    PetscPrintf(PETSC_COMM_WORLD,"Norm of error %g, Iterations %d\n",norm,its);
  else 
    PetscPrintf(PETSC_COMM_WORLD,"Norm of error < 1.e-12, Iterations %d\n",its);
    */

  ierr = MatMult(A,x,u); CHKERRA(ierr);
  ierr = VecAXPY(&neg_one,b,u); CHKERRA(ierr);
  ierr  = VecNorm(b,NORM_2,&bnorm); CHKERRA(ierr);
  ierr  = VecNorm(u,NORM_2,&rnorm); CHKERRA(ierr);

  termination_reason = "not-set";

  if      (ierrsolve == PETSC_ERR_MAT_LU_ZRPVT) 
    termination_reason = "zero-pvt";
  else if (ierrsolve == PETSC_ERR_MEM) 
    termination_reason = "outofmem";
  else if (ierrsolve != 0)
    termination_reason = "unknown";
  else if ( !strcmp(termination_reason,"not-set") && 
	    rnorm/bnorm > 1e5 )
    termination_reason = "div";
  else if ( !strcmp(termination_reason,"not-set") && 
	    rnorm/bnorm <= 1e-10 )
    termination_reason = "r-converged";
  else if ( !strcmp(termination_reason,"not-set") && 
	    rnorm <= 1e-100)
    termination_reason = "a-converged";
  else if ( !strcmp(termination_reason,"not-set") && its < 0 )
    termination_reason = "div";

  printf("iterations=%d\n",its);
  printf("residual norm = %lg\nrelative residual norm = %lg\noutcome: %s\n",
	 rnorm,rnorm/bnorm,termination_reason);

  /* 
     Free work space.  All PETSc objects should be destroyed when they
     are no longer needed.
  */
  ierr = VecDestroy(x); CHKERRA(ierr); ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = VecDestroy(b); CHKERRA(ierr); ierr = MatDestroy(A); CHKERRA(ierr);
  ierr = SLESDestroy(sles); CHKERRA(ierr);

  /*
     Always call PetscFinalize() before exiting a program.  This routine
       - finalizes the PETSc libraries as well as MPI
       - provides summary and diagnostic information if certain runtime
         options are chosen (e.g., -log_summary).
  */
  PetscFinalize();
  return 0;
}

int read_ijv(char *filename, Mat* A) {
  FILE *f_in;
  int ierr;
  int *colptr, *rowind, *count;
  int n, m, nnz, items, iscomplex; 
  double v,tmp;
  int i, j, l, k, last_j;

  fprintf(stderr,"reading matrix %s\n",filename);
  f_in = fopen(filename, "r"); 
  m=n=nnz=0;
  while (!feof(f_in)) {
    items = fscanf(f_in,"%d %d %lg",&i,&j,&v);
    if (items != 3) break;
    nnz++;
    if (i > m) m = i;
    if (j > n) n = j;
  }
  fprintf(stderr,"reading matrix; m=%d n=%d nnz=%d\n",m,n,nnz);
  fclose(f_in);
    
  ierr = MatCreate(PETSC_COMM_WORLD,m,n,A); CHKERRA(ierr);

  f_in = fopen(filename, "r"); 

  while (!feof(f_in)) {
    items = fscanf(f_in,"%d %d %lg",&i,&j,&v);
    if (items < 3) break;

    j--;
    i--;

    ierr = MatSetValues(*A,1,&i,1,&j,&v,INSERT_VALUES); CHKERRA(ierr);
  }

  ierr = MatAssemblyBegin(*A,MAT_FINAL_ASSEMBLY); CHKERRA(ierr);
  ierr = MatAssemblyEnd(*A,MAT_FINAL_ASSEMBLY); CHKERRA(ierr);

  fclose(f_in);

  fprintf(stderr,"done reading %s\n",filename);

  return n;
}
