!                  Kernel Program of PLOTTER
! Author: Masao Kodama
! E-mail: mkodama@mable.ne.jp
! Version: 2.1
! Date: March 18, 2009
MODULE plotter
REAL,PARAMETER:: fact=2.845,pi=3.14159265359   !  1mm=2.845pt
REAL,PARAMETER:: alnb2=3*fact,alnb1=alnb2*.7   !  broken line
REAL,PARAMETER:: alnd2=1.5*fact,alnd1=alnd2*.5 !  dotted line
REAL,PARAMETER:: alnc4=8*fact,alnc1=alnc4*.7,alnc2=alnc4*.8, &
                 alnc3=alnc4*.9, &             !  chain  line
                 const=80*fact ! 80=the initial value of xcomax and ycomax
REAL:: x00d=0,y00d=0,xcomax1=const,ycomax1=const, &
       xcomaxd=const,ycomaxd=const,heightd=4.,theta_deg=0
! x00d=x00*fact         y00d=y00*fact
! xcomax1=xcomax*fact   ycomax1=ycomax*fact
! xcomaxd=xcomax1+x00d  ycomaxd=ycomax+y00d
! heightd=the height of characters in unit mm.
! theta_deg: the slant angle in degrees.
INTEGER:: line_typed=1, &  ! line_type: the kind of lines
     iclipd=0,          &  ! iclipd: clip or non-clip
     nxd=0,nyd=0,       &  ! nxd, nyd: the internal position of character.
     iopen=-8              ! When iopen<0, an output file is not open.
END MODULE plotter

SUBROUTINE pfnbegin(char)
! Section 2.1 of the manual "Computer Graphics PLOTTER"
! Beginning Plotter
USE plotter,ONLY: iopen
IMPLICIT NONE
CHARACTER(LEN=*),INTENT(IN):: char
IF(iopen > 0) THEN
  PRINT *,'An error in SUBROUTINE pfnbegin of Plotter'
  PRINT *,'SUBROUTINE pfnbegin has been already invoked.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
OPEN(88,STATUS='REPLACE',FILE=char//'.ps')
iopen=2
END SUBROUTINE pfnbegin

SUBROUTINE pfnend
! Section 2.2 of the manual "Computer Graphics PLOTTER"
! Ending Plotter
! Closing the Postscript file for Plotter
USE plotter,ONLY: iopen
IMPLICIT NONE
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pfnend of Plotter'
  PRINT *,'SUBROUTINE pfnend has been already invoked,'
  PRINT *,'or SUBROUTINE pfnbegin must be invoked before SUBROUTINE pfnend.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
WRITE(88,*) ' showpage '
CLOSE(88,STATUS='KEEP')
iopen=-8
END SUBROUTINE pfnend

SUBROUTINE pfsorigin(x00,y00)
! Section 2.3 of the manual "Computer Graphics PLOTTER"
! Setting the origin
USE plotter,ONLY: fact,x00d,y00d,xcomax1,ycomax1,xcomaxd,ycomaxd
IMPLICIT NONE
REAL,INTENT(IN):: x00,y00
x00d=x00*fact;  y00d=y00*fact
xcomaxd=xcomax1+x00d;  ycomaxd=ycomax1+y00d
END SUBROUTINE pfsorigin

SUBROUTINE pfscolor(redc,greenc,bluec)
! Section 2.4 of the manual "Computer Graphics PLOTTER"
! Setting the color
! Argument redc indicates the strength of red.
! Argument greenc indicates the strength of green.
! Argument bluec indicates the strength of blue.
! The color of a combination of redc, greenc and bluec is determined by the 
! additive combination of colors.
! Default values of redc, greenc and bluec are 0.
USE plotter,ONLY: iopen
IMPLICIT NONE
REAL,INTENT(IN):: redc,greenc,bluec
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pfscolor of Plotter'
  PRINT *,'SUBROUTINE pfscolor has been already invoked,'
  PRINT *,'or SUBROUTINE pfnbegin must be invoked before SUBROUTINE pfscolor.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(MAX(redc,greenc,bluec) > 100.01) THEN
  PRINT *,'An error in SUBROUTINE pfscolor(redc,greenc,bluec) of Plotter'
  PRINT 11,'redc  =',redc
  PRINT 11,'greenc=',greenc
  PRINT 11,'bluec =',bluec
  PRINT *,'It is necessary that MAX(redc,greenc,bluec)<=100.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(MIN(redc,greenc,bluec) < -0.01) THEN
  PRINT *,'An error in SUBROUTINE pfscolor(redc,greenc,bluec) of Plotter'
  PRINT 11,'redc  =',redc
  PRINT 11,'greenc=',greenc
  PRINT 11,'bluec =',bluec
  PRINT *,'It is necessary that MIN(redc,greenc,bluec)>=0.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
11 FORMAT(1X,A,F6.1)
WRITE(88,8) redc*0.01,greenc*0.01,bluec*0.01,' setrgbcolor'
8 FORMAT(3F8.2,A)
END SUBROUTINE pfscolor

SUBROUTINE plstype(line_type)
! Section 2.5 of the manual "Computer Graphics PLOTTER"
! Setting the line type
! line_type=1: solid line     line_type=2: broken line
! line_type=3: dotted line    line_type=4: chain line
USE plotter,ONLY: line_typed
IMPLICIT NONE
INTEGER,INTENT(IN):: line_type
IF(line_type<=0 .OR. line_type>=5) THEN
  PRINT *,'An error in SUBROUTINE plstype(line_type) of Plotter'
  PRINT *,'line_type=',line_type
  PRINT *,'The value of argument line_type is invalid.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
line_typed=line_type
END SUBROUTINE plstype

SUBROUTINE plswidth(width)
! Section 2.6 of the manual "Computer Graphics PLOTTER"
! Setting the line width
USE plotter,ONLY: fact
IMPLICIT NONE
REAL,INTENT(IN):: width
IF(width < 0) THEN
  PRINT *,'An error in SUBROUTINE plswidth(width) of Plotter'
  PRINT '(A,F6.1)',' width=',width
  PRINT *,'The argument width must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(width > 20) THEN
  PRINT *,'An error in SUBROUTINE plswidth(width) of Plotter'
  PRINT '(A,F6.1)',' width=',width
  PRINT *,'The argument width must be less than 20.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
WRITE(88,*) width*fact,' setlinewidth'
END SUBROUTINE plswidth

SUBROUTINE plsframe(xcomax,ycomax)
! Section 2.7 of the manual "Computer Graphics PLOTTER"
! Setting the clipping frame
! The clipping frame is a rectangle. 
! The vertexes of the rectangle are at (0., 0.), (xcomax, 0.), (xcomax, ycomax)
! and (0., ycomax).
USE plotter,ONLY: fact,x00d,y00d,xcomax1,ycomax1,xcomaxd,ycomaxd
IMPLICIT NONE
REAL,INTENT(IN):: xcomax,ycomax
IF(xcomax <= 0) THEN
  PRINT *,'An error in SUBROUTINE plsframe(xcomax,ycomax) of Plotter'
  PRINT '(A,F6.1)',' xcomax=',xcomax
  PRINT *,'The argument xcomax must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(ycomax <= 0) THEN
  PRINT *,'An error in SUBROUTINE plsframe(xcomax,ycomax) of Plotter'
  PRINT '(A,F6.1)',' ycomax=',ycomax
  PRINT *,'The argument ycomax must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
xcomax1=xcomax*fact;   ycomax1=ycomax*fact
xcomaxd=xcomax1+x00d;  ycomaxd=ycomax1+y00d
END SUBROUTINE plsframe

SUBROUTINE plsclip(iclip)
! Section 2.8 of the manual "Computer Graphics PLOTTER"
! Setting the clipping indicator
! iclip=0: non_clipping,   iclip=1: clipping
USE plotter,ONLY: iclipd
IMPLICIT NONE
INTEGER,INTENT(IN):: iclip
IF(iclip/=0 .AND. iclip/=1) THEN
  PRINT *,'An error in SUBROUTINE plsclip(iclip) of Plotter'
  PRINT *,'iclip=',iclip
  PRINT *,'The value of iclip must be 0 or 1.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
iclipd=iclip
END SUBROUTINE plsclip

SUBROUTINE pldline(nn,dxx,dyy)
! Section 2.9 of the manual "Computer Graphics PLOTTER"
! Drawing a polygonal line
! The internal procedures: gpl1, gpl2, gpl3, gpl4 and gsgmnt1.
USE plotter,ONLY: fact,line_typed,x00d,y00d,alnb1,alnb2,alnc1,alnc2,alnc3, &
  alnc4,alnd1,alnd2,iopen
IMPLICIT NONE
REAL,INTENT(IN):: dxx(*),dyy(*)
INTEGER,INTENT(IN):: nn
REAL:: dx1(nn),dy1(nn)
INTEGER:: i
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldline of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldline.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(nn <= 1) THEN
  PRINT *,'An error in SUBROUTINE pldline(nn,dxx,dyy) of Plotter'
  PRINT *,'nn=',nn
  PRINT *,'The argument nn must be larger than 1.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
DO i=1,nn
  IF(ABS(dxx(i)) > 30000) THEN
    PRINT *,'An error in SUBROUTINE pldline(nn,dxx,dyy) of Plotter'
    WRITE(*,11) 'dxx(',i,')=',dxx(i)
    11 FORMAT(1X,A,I4,A,2G14.7)
    WRITE(*,*) 'The values of ABS(dxx(i)) must be less than 30000.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  IF(ABS(dyy(i)) > 30000) THEN
    PRINT *,'An error in SUBROUTINE pldline(nn,dxx,dyy) of Plotter'
    WRITE(*,11) 'dyy(',i,')=',dyy(i)
    WRITE(*,*) 'The values of ABS(dyy(i)) must be less than 30000.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  dx1(i)=fact*dxx(i)+x00d;  dy1(i)=fact*dyy(i)+y00d
ENDDO
SELECT CASE(line_typed)
  CASE(1);  CALL gpl1   ! solid line
  CASE(2);  CALL gpl2   ! broken line
  CASE(3);  CALL gpl3   ! dotted line
  CASE(4);  CALL gpl4   ! chain line
  CASE DEFAULT;  STOP 'Error in pldline of Plotter'
END SELECT
CONTAINS

SUBROUTINE gpl1
! An internal procedure of SUBROUTINE pldline
! Solid line
! The variables used in common with the host procedure: nn, dx1 and dy1
DO i=2,nn
  CALL gsgmnt1(dx1(i-1),dy1(i-1),dx1(i),dy1(i))
ENDDO
END SUBROUTINE gpl1

SUBROUTINE gpl2
! An internal procedure of SUBROUTINE pldline
! Broken line
! The variables used in common with the host procedure: nn, dx1 and dy1
REAL:: ast0,ast1,b1,compx,compy,posit,x1,y1,x2,y2
INTEGER:: i
posit=0
DO i=2,nn
  ast0=0
  ast1=SQRT((dx1(i)-dx1(i-1))**2+(dy1(i)-dy1(i-1))**2)
  IF(ast1 <= 1E-8) THEN
    compx=0;  compy=0
  ELSE
    compx=(dx1(i)-dx1(i-1))/ast1;  compy=(dy1(i)-dy1(i-1))/ast1
  ENDIF
  DO
    x1=dx1(i-1)+ast0*compx;  y1=dy1(i-1)+ast0*compy
    IF(ast1-ast0 <= alnb2-posit) THEN
      IF(posit < alnb1) THEN
        b1=MIN(ast1-ast0,alnb1-posit)+ast0
        x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
        CALL gsgmnt1(x1,y1,x2,y2)
      ENDIF
      posit=posit+ast1-ast0;  ast0=0;  EXIT
    ELSE
    IF(posit < alnb1) THEN
      b1=ast0+alnb1-posit
      x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
      CALL gsgmnt1(x1,y1,x2,y2)
    ENDIF
    ast0=ast0+alnb2-posit;  posit=0
    ENDIF
  ENDDO
ENDDO
END SUBROUTINE gpl2

SUBROUTINE gpl3
! An internal procedure of SUBROUTINE pldline
! Dotted line
! The variables used in common with the host procedure: nn, dx1 and dy1
REAL:: ast0,ast1,posit,compx,compy,b1,x1,y1,x2,y2
posit=0
DO i=2,nn
  ast0=0
  ast1=SQRT((dx1(i)-dx1(i-1))**2+(dy1(i)-dy1(i-1))**2)
  IF(ast1 <= 1e-8) THEN
    compx=0;  compy=0
  ELSE
    compx=(dx1(i)-dx1(i-1))/ast1;  compy=(dy1(i)-dy1(i-1))/ast1
  ENDIF
  DO
    x1=dx1(i-1)+ast0*compx;  y1=dy1(i-1)+ast0*compy
    IF(ast1-ast0 <= alnd2-posit) THEN
      IF(posit < alnd1) THEN
        b1=MIN(ast1-ast0,alnd1-posit)+ast0
        x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
        CALL gsgmnt1(x1,y1,x2,y2)
      ENDIF
      posit=posit+ast1-ast0;  ast0=0;  EXIT
    ELSE
      IF(posit < alnd1) THEN
        b1=ast0+alnd1-posit
        x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
        CALL gsgmnt1(x1,y1,x2,y2)
      ENDIF
      ast0=ast0+alnd2-posit;  posit=0
    ENDIF
  ENDDO
ENDDO
END SUBROUTINE gpl3

SUBROUTINE gpl4
! An internal procedure of SUBROUTINE pldline
! Chain line
! The variables used in common with the host procedure: nn, dx1 and dy1
REAL:: ast0,ast1,b1,compx,compy,posit,x1,y1,x2,y2,x3,y3,x4,y4
posit=0
DO i=2,nn
  ast0=0
  ast1=SQRT((dx1(i)-dx1(i-1))**2+(dy1(i)-dy1(i-1))**2)
  IF(ast1 <= 1E-8) THEN
    compx=0;  compy=0
  ELSE
    compx=(dx1(i)-dx1(i-1))/ast1;  compy=(dy1(i)-dy1(i-1))/ast1
  ENDIF
  DO
    x1=dx1(i-1)+ast0*compx;  y1=dy1(i-1)+ast0*compy
    IF(ast1-ast0 <= alnc4-posit) THEN
      IF(posit < alnc1) THEN
        b1=MIN(ast1-ast0,alnc1-posit)+ast0
        x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
        CALL gsgmnt1(x1,y1,x2,y2)
      ENDIF
      IF(posit < alnc3) THEN
        b1=MIN(ast1-ast0,MAX(alnc2-posit,0.))+ast0
        x3=dx1(i-1)+b1*compx;  y3=dy1(i-1)+b1*compy
        b1=MIN(ast1-ast0,alnc3-posit)+ast0
        x4=dx1(i-1)+b1*compx;  y4=dy1(i-1)+b1*compy
        CALL gsgmnt1(x3,y3,x4,y4)
      ENDIF
      posit=posit+ast1-ast0;  ast0=0;  EXIT
    ELSE
      IF(posit < alnc1) THEN
        b1=ast0+alnc1-posit
        x2=dx1(i-1)+b1*compx;  y2=dy1(i-1)+b1*compy
        CALL gsgmnt1(x1,y1,x2,y2)
      ENDIF
      IF(posit < alnc3) THEN
        b1=MIN(ast1-ast0,MAX(alnc2-posit,0.))+ast0
        x3=dx1(i-1)+b1*compx;  y3=dy1(i-1)+b1*compy
        b1=MIN(ast1-ast0,alnc3-posit)+ast0
        x4=dx1(i-1)+b1*compx;  y4=dy1(i-1)+b1*compy
        CALL gsgmnt1(x3,y3,x4,y4)
      ENDIF
      ast0=ast0+alnc4-posit;   posit=0
    ENDIF
  ENDDO
ENDDO
END SUBROUTINE gpl4

SUBROUTINE gsgmnt1(x1,y1,x2,y2)
! An internal procedure of SUBROUTINE pldline
! Drawing a solid segment independently of the values of iclip and line_type.
! (x1,y1), (x2,y2)=both ends of segment
! The variable used in common with the host procedure: x00d and y00d
USE plotter,ONLY: iclipd,xcomaxd,ycomaxd
REAL,INTENT(IN):: x1,y1,x2,y2
REAL:: a1,x2d,y2d,x1d,y1d,xs,ys
INTEGER:: icut
x1d=x1;  y1d=y1;  x2d=x2;  y2d=y2
IF(iclipd == 0) THEN
  WRITE(88,101) 'newpath ',x1,y1,' moveto ',x2,y2,' lineto stroke'
ELSE
  icut=0
  IF((x00d-x1d)*(x00d-x2d) < 0) THEN
    ys=(y2d-y1d)*(x00d-x1d)/(x2d-x1d)+y1d
    IF((ys-y00d)*(ys-ycomaxd) < 0) THEN
      IF(x00d < x1d) THEN
        a1=x2d;  x2d=x1d;  x1d=a1
        a1=y2d;  y2d=y1d;  y1d=a1
      ENDIF
      x1d=x00d;  y1d=ys;  icut=1
    ENDIF
  ENDIF
  IF((xcomaxd-x1d)*(xcomaxd-x2d) < 0) THEN
    ys=(y2d-y1d)*(xcomaxd-x1d)/(x2d-x1d)+y1d
    IF((ys-y00d)*(ys-ycomaxd) < 0) THEN
      IF(xcomaxd > x1d) THEN
        a1=x2d;  x2d=x1d;  x1d=a1
        a1=y2d;  y2d=y1d;  y1d=a1
      ENDIF
      x1d=xcomaxd;  y1d=ys;  icut=1
    ENDIF
  ENDIF
  IF((y00d-y1d)*(y00d-y2d) < 0) THEN
    xs=(x2d-x1d)*(y00d-y1d)/(y2d-y1d)+x1d
    IF((xs-x00d)*(xs-xcomaxd) < 0) THEN
      IF(y00d < y1d) THEN
        a1=x2d;  x2d=x1d;  x1d=a1
        a1=y2d;  y2d=y1d;  y1d=a1
      ENDIF
      x1d=xs;  y1d=y00d;  icut=1
    ENDIF
  ENDIF
  IF((ycomaxd-y1d)*(ycomaxd-y2d) < 0) THEN
    xs=(x2d-x1d)*(ycomaxd-y1d)/(y2d-y1d)+x1d
    IF((xs-x00d)*(xs-xcomaxd) < 0) THEN
      IF(ycomaxd > y1d) THEN
        a1=x2d;  x2d=x1d;  x1d=a1
        a1=y2d;  y2d=y1d;  y1d=a1
      ENDIF
      x1d=xs;  y1d=ycomaxd;  icut=1
    ENDIF
  ENDIF
  IF(icut==0 .AND. (x00d-x1d)*(xcomaxd-x1d)<=0 &
    .AND. (y00d-y1d)*(ycomaxd-y1d)<=0 .AND. &
    (x00d-x2d)*(xcomaxd-x2d)<=0 .AND. (y00d-y2d)*(ycomaxd-y2d)<=0) icut=1
  IF(icut == 1) &
      WRITE(88,101) 'newpath',x1d,y1d,' moveto',x2d,y2d,' lineto stroke'
ENDIF
101 FORMAT(A,2F10.2,A,2F10.2,A)
END SUBROUTINE gsgmnt1

END SUBROUTINE pldline

SUBROUTINE pldsgmnt(x1,y1,x2,y2)
! Section 2.10 of the manual "Computer Graphics PLOTTER"
! Drawing a segment
! (x1,y1) and (x2,y2)=both ends of the segment
! The subroutine invoked by this subroutine: pldline.
USE plotter,ONLY: iopen
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,x2,y2
REAL:: dx(2),dy(2)
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldsgmnt of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldsgmnt.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
dx(1)=x1;  dy(1)=y1
dx(2)=x2;  dy(2)=y2
CALL pldline(2,dx,dy)
END SUBROUTINE pldsgmnt

SUBROUTINE pldarrow(x1,y1,x2,y2)
! Section 2.11 of the manual "Computer Graphics PLOTTER"
! Drawing an arrowhead
! The arrowhead points from (x1,y1) to (x2,y2)
! The subroutine invoked by this subroutine: pldsgmnt.
USE plotter,ONLY: iopen,line_typed
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,x2,y2
REAL,PARAMETER:: a1=2.,th2=.51
REAL:: th
INTEGER:: i
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldarrow of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldarrow.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(ABS(x1-x2)+ABS(y1-y2) <= 0) THEN
  PRINT *,'An error in SUBROUTINE pldarrow(x1,y1,x2,y2) of Plotter'
  PRINT '(A,F6.1)',' x1=x2=',x1
  PRINT '(A,F6.1)',' y1=y2=',y1
  PRINT *,'It is necessary that ABS(x1-x2)+ABS(y1-y2)>0.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
th=ATAN2(y1-y2,x1-x2)
i=line_typed;   line_typed=1
CALL pldsgmnt(x2+a1*COS(th+th2),y2+a1*SIN(th+th2),x2,y2)
CALL pldsgmnt(x2+a1*COS(th-th2),y2+a1*SIN(th-th2),x2,y2)
line_typed=i
END SUBROUTINE pldarrow

SUBROUTINE pldcrcl(x1,y1,rr)
! Section 2.12 of the manual "Computer Graphics PLOTTER"
! Drawing a circle
! (x1,y1)=the center of the circle,    rr=the radius of the circle
USE plotter,ONLY: iopen,fact,x00d,y00d,pi,line_typed, &
  alnb1,alnb2,alnc1,alnc2,alnc3,alnc4,alnd1,alnd2
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,rr
REAL:: arr,ax1,ay1,deg1,deg2,deg3,deg4,deg5,deg6,deg7,deg8
INTEGER:: i,nn
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldcrcl of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldcrcl.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(rr <= 0) THEN
  PRINT *,'An error in SUBROUTINE pldcrcl(x1,y1,rr) of Plotter'
  PRINT '(A,F6.1)',' rr=',rr
  PRINT *,'The argument rr must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
ax1=fact*x1+x00d;  ay1=fact*y1+y00d;  arr=fact*rr
SELECT CASE(line_typed)
  CASE(1);  CALL pldcrclarc(x1,y1,rr,0.,2*pi)       ! solid line
  CASE(2);  nn=NINT(2*pi*arr/alnb2);  nn=MAX(1,nn)  ! broken line
    ! alnb2=3*fact,  alnb1=alnb2*.7:                  broken line
    deg2=360./nn;  deg1=deg2*(alnb1/alnb2)
    DO i=1,nn
      deg5=deg2*i;  deg6=deg5+deg1
      WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
      100 FORMAT(A,5F9.2,A)
    ENDDO
  CASE(3);  nn=NINT(2*pi*arr/alnd2);  nn=MAX(1,nn)  ! dotted line
    ! alnd2=1.5*fact,  alnd1=alnd2*.5:                dotted line
    deg2=360./nn;  deg1=deg2*(alnd1/alnd2)
    DO i=1,nn
      deg5=deg2*i;  deg6=deg5+deg1
      WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
    ENDDO
  CASE(4);  nn=NINT(2*pi*arr/alnc4);  nn=MAX(1,nn)  !  chain line
    ! alnc4=8*fact, alnc1=alnc4*.7, alnc2=alnc4*.8, alnc3=alnc4*.9:  chain line
    deg4=360./nn;  deg1=deg4*(alnc1/alnc4);  deg2=deg4*(alnc2/alnc4)
    deg3=deg4*(alnc3/alnc4)
    DO i=1,nn
      deg5=deg4*i;  deg6=deg5+deg1
      WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
      deg7=deg5+deg2;  deg8=deg5+deg3
      WRITE(88,100) 'newpath ',ax1,ay1,arr,deg7,deg8,' arc stroke'
    ENDDO
  CASE DEFAULT;  STOP 'Error in pldcrcl of Plotter'
END SELECT
END SUBROUTINE pldcrcl

SUBROUTINE pldcrclarc(x1,y1,rr,phi1,phi2)
! Section 2.13 of the manual "Computer Graphics PLOTTER"
! Drawing a circular arc
! (x1,y1)=the center of circular arc,    rr=the radius of circular arc
! phi1=the starting angle,               phi2=the arriving angle
! It is necessary that phi1<phi2<phi1+2*pi.
USE plotter,ONLY: iopen,fact,x00d,y00d,pi,line_typed, &
  alnb1,alnb2,alnc1,alnc2,alnc3,alnc4,alnd1,alnd2
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,rr,phi1,phi2
REAL:: arr,ax1,ay1,adeg1,adeg2,deg1,deg,deg2,deg3,deg4,deg5,deg6,deg7,deg8, &
  degtotal
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldcrclarc of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldcrclarc.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(rr <= 0) THEN
  PRINT *,'An error in SUBROUTINE pldcrclarc(x1,y1,rr,phi1,phi2) of Plotter'
  PRINT '(A,F6.1)',' rr=',rr
  PRINT *,'The argument rr must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(phi2 < phi1) THEN
  PRINT *,'An error in SUBROUTINE pldcrclarc(x1,y1,rr,phi1,phi2) of Plotter'
  PRINT '(A,F6.1)',' phi1=',phi1
  PRINT '(A,F6.1)',' phi2=',phi2
  PRINT *,'Angle phi2 must be larger than angle phi1.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(phi2 > phi1+2*pi) THEN
  PRINT *,'An error in SUBROUTINE pldcrclarc(x1,y1,rr,phi1,phi2) of Plotter'
  PRINT '(A,F6.1)',' phi1=',phi1
  PRINT '(A,F6.1)',' phi2=',phi2
  PRINT *,'Angle phi2 must be less than angle phi1+2*pi.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(ABS(phi1)>4*pi .OR. ABS(phi2)>4*pi) THEN
  PRINT *,'An error in SUBROUTINE pldcrclarc of Plotter'
  PRINT '(A,F6.1)',' phi1=',phi1
  PRINT '(A,F6.1)',' phi2=',phi2
  PRINT *,'It must be satisfied that ABS(phi1)<4*pi and ABS(phi2)<4*pi.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
ax1=fact*x1+x00d;  ay1=fact*y1+y00d;  arr=fact*rr
adeg1=phi1*180./pi;  adeg2=phi2*180./pi
SELECT CASE(line_typed)
  CASE(1)
    WRITE(88,100) 'newpath ',ax1,ay1,arr,adeg1,adeg2,' arc stroke'
  CASE(2)    ! alnb2=3*fact, alnb1=alnb2*.7:     broken line
    degtotal=adeg2-adeg1;  deg=0;  deg2=(alnb2/arr)*180/pi
    deg1=deg2*(alnb1/alnb2)
    DO
      IF(degtotal-deg > deg2) THEN
        deg5=adeg1+deg;  deg6=deg5+deg1
        WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
      ELSE;  EXIT
      ENDIF
      deg=deg+deg2
    ENDDO
    deg5=adeg1+deg
    deg6=deg5+MIN(deg1,degtotal-deg)
    WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
  CASE(3)    ! alnd2=1.5*fact, alnd1=alnd2*.5:   dotted line
    degtotal=adeg2-adeg1;  deg=0;  deg2=(alnd2/arr)*180/pi
    deg1=deg2*(alnd1/alnd2)
    DO
      IF(degtotal-deg > deg2) THEN
        deg5=adeg1+deg;  deg6=deg5+deg1
        WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
      ELSE;  EXIT
      ENDIF
      deg=deg+deg2
    ENDDO
    deg5=adeg1+deg
    deg6=deg5+MIN(deg1,degtotal-deg)
    WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
  CASE(4)
    ! alnc4=8*fact, alnc1=alnc4*.7, alnc2=alnc4*.8, alnc3=alnc4*.9: chain line
    degtotal=adeg2-adeg1;  deg=0;  deg4=(alnc4/arr)*180/pi;
    deg1=deg4*(alnc1/alnc4);  deg2=deg4*(alnc2/alnc4)
    deg3=deg4*(alnc3/alnc4)
    DO
      IF(degtotal-deg > deg4) THEN
        deg5=adeg1+deg;  deg6=deg5+deg1
        WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
        deg7=deg5+deg2;  deg8=deg5+deg3
        WRITE(88,100) 'newpath ',ax1,ay1,arr,deg7,deg8,' arc stroke'
      ELSE;  EXIT
      ENDIF
      deg=deg+deg4
    ENDDO
    deg5=adeg1+deg
    deg6=deg5+MIN(deg1,degtotal-deg)
    WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
    IF(degtotal-deg > deg2) THEN
      deg5=adeg1+deg+deg2
      deg6=adeg1+deg+MIN(deg3,adeg2-deg)
      WRITE(88,100) 'newpath ',ax1,ay1,arr,deg5,deg6,' arc stroke'
    ENDIF
  CASE DEFAULT;  STOP 'Error in pldcrclarc of Plotter'
END SELECT
100 FORMAT(A,5F9.2,A)
END SUBROUTINE pldcrclarc

SUBROUTINE pldarcarrow(x1,y1,rr,phi1,phi2)
! Section 2.14 of the manual "Computer Graphics PLOTTER"
! Drawing an arrowhead on a circular arc
! (x1,y1)=the center of the circular arc
! phi2=the angle of the position of the point of the arrowhead
! If phi1<=phi2, the arrowhead points anti-clockwise. 
! If phi1> phi2, the arrowhead points clockwise. 
! It must be satisfied that ABS(ph1)<4*pi and ABS(phi2)<4*pi.
! The subroutine invoked by this subroutine: pldarrow.
USE plotter,ONLY: iopen,pi
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,rr,phi1,phi2
REAL:: x3,y3,phi5
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldarcarrow of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldarcarrow.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(rr <= 0) THEN
  PRINT *,'An error in SUBROUTINE pldarcarrow(x1,y1,rr,phi1,phi2) of Plotter'
  PRINT '(A,F6.1)',' rr=',rr
  PRINT *,'The argument rr must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(ABS(phi1)>4*pi .OR. ABS(phi2)>4*pi) THEN
  PRINT *,'An error in SUBROUTINE pldarcarrow(x1,y1,rr,phi1,phi2) of Plotter'
  PRINT '(A,F6.1)',' phi1=',phi1
  PRINT '(A,F6.1)',' phi2=',phi2
  PRINT *,'It must be satisfied that ABS(phi1)<4*pi and ABS(phi2)<4*pi.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
x3=x1+rr*COS(phi2);  y3=y1+rr*SIN(phi2)
phi5=phi2-SIGN(1.5707963,phi2-phi1)
CALL pldarrow(x3+COS(phi5),y3+SIN(phi5),x3,y3)
END SUBROUTINE pldarcarrow

SUBROUTINE pldrctgl(x1,y1,x2,y2)
! Section 2.15 of the manual "Computer Graphics PLOTTER"
! Drawing a rectangle
! The vertexes of the rectangular are (x1,y1), (x2,y1), (x2,y2) and (x1,y2).
! The subroutine invoked by this subroutine: pldline.
USE plotter,ONLY: iopen
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,x2,y2
REAL:: dx(5),dy(5)
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldrctgl of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldrctgl.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
dx(1)=x1;  dy(1)=y1
dx(2)=x2;  dy(2)=y1
dx(3)=x2;  dy(3)=y2
dx(4)=x1;  dy(4)=y2
dx(5)=x1;  dy(5)=y1
CALL pldline(5,dx,dy)
END SUBROUTINE pldrctgl

SUBROUTINE pldcross(x1,y1,rr)
! Section 2.16 of the manual "Computer Graphics PLOTTER"
! Drawing a cross
! (x1,y1)=the center of the cross,   rr=the radius of the cross
! The subroutine invoked by this subroutine: pldsgmnt.
USE plotter,ONLY: iopen,line_typed
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,rr
REAL:: a1
INTEGER:: i
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldcross of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldcross.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(rr <= 0) THEN
  PRINT *,'An error in SUBROUTINE pldcross(x1,y1,rr) of Plotter'
  PRINT '(A,F6.1)',' rr=',rr
  PRINT *,'The argument rr must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
i=line_typed;  line_typed=1;  a1=.7071*rr
CALL pldsgmnt(x1+a1,y1+a1,x1-a1,y1-a1)
CALL pldsgmnt(x1-a1,y1+a1,x1+a1,y1-a1)
line_typed=i
END SUBROUTINE pldcross

SUBROUTINE pldhatching(nn,dxx,dyy,sp,theta)
! Section 2.17 of the manual "Computer Graphics PLOTTER"
! Drawing hatching
! nn=number of vertexes of the polygon,  dxx,dyy=coordinates of vertexes
! sp=the interval of the parallel lines
! theta=the slant angle of the parallel lines
! The internal procedure: sleq2.
! The subroutine invoked by this subroutine: pldsgmnt.
USE plotter,ONLY: iopen,pi
IMPLICIT NONE
REAL,INTENT(IN):: dxx(*),dyy(*),sp,theta
INTEGER,INTENT(IN):: nn
REAL,DIMENSION(nn):: daa,dbb,dcc
REAL:: a,as,fx,fy,fx1,fy1
INTEGER,PARAMETER:: imax=20
! imax: the maximum number of the cross points between a parallel line and
!       the polygon.
REAL:: da(2,2),db(2),dx1(imax),dy1(imax),dcomp(imax)
INTEGER:: i,ii,info,jj,j1
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pldhatching of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pldhatching.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(nn <= 2) THEN
  PRINT *,'An error in SUBROUTINE pldhatching(nn,dx,dy,sp,theta) of Plotter'
  PRINT *,'nn=',nn
  PRINT *,'It is not permitted that nn<=2.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(sp < 0.) THEN
  PRINT *,'An error in SUBROUTINE pldhatching(nn,dx,dy,sp,theta) of Plotter'
  PRINT '(A,F6.1)',' sp=',sp
  PRINT *,'It is necessary that sp>0.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(ABS(theta) > pi) THEN
  PRINT *,'An error in SUBROUTINE pldhatching(nn,dx,dy,sp,theta) of Plotter'
  PRINT '(A,F6.1)',' theta=',theta
  PRINT *,'It is necessary that ABS(theta)<pi'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
! To obtain the equation of the polygon.
DO i=1,nn
  ii=i+1;  IF(ii == nn+1) ii=1
  db(1)=1.;  da(1,1)=dxx(i);   da(1,2)=dyy(i)
  db(2)=1.;  da(2,1)=dxx(ii);  da(2,2)=dyy(ii)
  CALL sleq2(info)
  IF(info == 0) THEN
    daa(i)=db(1);  dbb(i)=db(2);  dcc(i)=1.
  ELSE
    IF(ABS(dxx(I))+ABS(dyy(i)) > ABS(dxx(ii))+ABS(dyy(ii))) THEN
      daa(i)=-dyy(i);   dbb(i)=dxx(i)
    ELSE
      daa(i)=-dyy(ii);  dbb(i)=dxx(ii)
    ENDIF
    dcc(i)=0.
  ENDIF
ENDDO
fx=-SIN(theta);  fy=COS(theta);  as=1E20
DO i=1,nn
  a=fx*dxx(i)+fy*dyy(i);  IF(a < as) as=a
ENDDO
fx1=COS(theta);  fy1=SIN(theta);  as=as+sp/2.
! To obtain cross points.
5 jj=0
DO i=1,nn
  ii=i+1;  IF(ii == nn+1) ii=1
  a=(fx*dxx(i)+fy*dyy(i)-as)*(fx*dxx(ii)+fy*dyy(ii)-as)
  IF(a < 0) THEN
    db(1)=dcc(i);  da(1,1)=daa(i);  da(1,2)=dbb(i)
    db(2)=as;      da(2,1)=fx;      da(2,2)=fy
    CALL sleq2(info)
    IF(info /= 0) STOP 'Error in pldhatching of Plotter'
    jj=jj+1
    IF(jj > imax) THEN
      PRINT *,'An error in SUBROUTINE pldhatching of Plotter'
      PRINT *,'The number of cross points between the polygon and a oblique'
      WRITE(*,8) ' line is larger than ',imax,'.'
      WRITE(*,8) ' It must be less than ',imax+1,'.'
      8 FORMAT(A,I2,A)
      PRINT *,'Stop of execution';  PRINT *;  STOP
    ENDIF
    dx1(jj)=db(1);  dy1(jj)=db(2)
    dcomp(jj)=fx1*dx1(jj)+fy1*dy1(jj)
  ENDIF
ENDDO
! Arrangement of the cross points
7 j1=0
DO i=1,jj-1
  IF(dcomp(i+1) > dcomp(i)) THEN
    a=dcomp(i+1);  dcomp(i+1)=dcomp(i);  dcomp(i)=a
    a=dx1(i+1);    dx1(i+1)=dx1(i);      dx1(i)=a
    a=dy1(I+1);    dy1(i+1)=dy1(i);      dy1(i)=a
    j1=j1+1
  ENDIF
ENDDO
IF(j1 >= 1) GOTO 7
DO i=1,jj,2
  CALL pldsgmnt(dx1(i),dy1(i),dx1(i+1),dy1(i+1))
ENDDO
as=as+sp
IF(jj > 0) GOTO 5
CONTAINS

SUBROUTINE sleq2(info)
! An internal procedure of SUBROUTINE pldhatching
! Solving 2-dimensional linear simultaneous equations.
! The variables used in common with the host procedure: da(2,2) and db(2)
INTEGER,INTENT(OUT):: info
REAL,DIMENSION(2):: db2
REAL:: dd
db2=db
dd=da(1,1)*da(2,2)-da(1,2)*da(2,1)
info=0; IF(ABS(dd) <= 1E-10) THEN;  info=10;  RETURN;  ENDIF
db(1)=( da(2,2)*db2(1)-da(1,2)*db2(2))/dd
db(2)=(-da(2,1)*db2(1)+da(1,1)*db2(2))/dd
END SUBROUTINE sleq2

END SUBROUTINE pldhatching

SUBROUTINE prprctgl(x1,y1,x2,y2)
! Section 2.18 of the manual "Computer Graphics PLOTTER"
! Painting a rectangular region
! The vertexes of the rectangular are (x1,y1), (x2,y1), (x2,y2) and (x1,y2).
! The subroutine invoked by this subroutine: prpplygn.
USE plotter,ONLY: iopen
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,x2,y2
REAL:: dx(4),dy(4)
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE prprctgl of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE prprctgl.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
dx(1)=x1;  dx(2)=x2;  dx(3)=x2;  dx(4)=x1
dy(1)=y1;  dy(2)=y1;  dy(3)=y2;  dy(4)=y2
CALL prpplygn(4,dx,dy)
END SUBROUTINE prprctgl

SUBROUTINE prpcrcl(x1,y1,rr)
! Section 2.19 of the manual "Computer Graphics PLOTTER"
! Painting a circular region
! (x1,y1)=the center of the circle,  rr=the radius of the circle
USE plotter,ONLY: iopen,fact,x00d,y00d
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1,rr
REAL:: x1d,y1d,rrd
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE prpcrcl of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE prpcrcl.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(rr <= 0) THEN
  PRINT *,'An error in SUBROUTINE prpcrcl(x1,y1,rr) of Plotter'
  PRINT '(A,F6.1)',' rr=',rr
  PRINT *,'It is necessary that rr>0.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
x1d=fact*x1+x00d;  y1d=fact*y1+y00d;  rrd=fact*rr
WRITE(88,10) 'newpath ',x1d,y1d,rrd,' 0 360 arc fill'
10 FORMAT(A,3F9.2,A)
END SUBROUTINE prpcrcl

SUBROUTINE prpplygn(nn,dxx,dyy)
! Section 2.20 of the manual "Computer Graphics PLOTTER"
! Painting a polygonal region
USE plotter,ONLY: iopen,fact,x00d,y00d
IMPLICIT NONE
INTEGER,INTENT(IN):: nn
REAL,INTENT(IN):: dxx(*),dyy(*)
REAL:: x1d,y1d
INTEGER:: i
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE prpplygn of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE prpplygn.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(nn <= 2) THEN
  PRINT *,'An error in SUBROUTINE prpplygn(nn,dxx,dyy) of Plotter'
  PRINT *,'nn=',nn
  PRINT *,'It is necessary that nn>=3.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
WRITE(88,*) 'newpath'
x1d=fact*dxx(1)+x00d;  y1d=fact*dyy(1)+y00d
WRITE(88,10) x1d,y1d,' moveto'
DO i=2,nn
  x1d=fact*dxx(i)+x00d;  y1d=fact*dyy(i)+y00d
  WRITE(88,10) x1d,y1d,' lineto'
ENDDO
10 FORMAT(2F9.2,A,2F9.2,A)
WRITE(88,*) 'fill'
END SUBROUTINE prpplygn

SUBROUTINE pcsheight(height)
! Section 2.21 of the manual "Computer Graphics PLOTTER"
! Setting the height of character patterns
USE plotter,ONLY: iopen,heightd
IMPLICIT NONE
REAL,INTENT(IN):: height
IF(height <= 0) THEN
  PRINT *,'An error in SUBROUTINE pcsheight(height) of Plotter'
  PRINT '(A,F6.1)',' height=',height
  PRINT *,'The argument height must be positive.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
IF(height > 200) THEN
  PRINT *,'An error in SUBROUTINE pcsheight(height) of Plotter'
  PRINT '(A,F6.1)',' height=',height
  PRINT *,'The argument height must be less than 200.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
heightd=height                      ! heightd: size of font in unit mm
END SUBROUTINE pcsheight

SUBROUTINE pcspstn(nx,ny)
! Section 2.22 of the manual "Computer Graphics PLOTTER"
! Setting the internal position of character patterns
USE plotter,ONLY: nxd,nyd
IMPLICIT NONE
INTEGER,INTENT(IN):: nx,ny
IF(nx<=-1 .OR. nx>=3) THEN
  PRINT *,'An error in SUBROUTINE pcspstn(nx,ny) of Plotter'
  PRINT *,'nx=',nx
  PRINT *,'The value of argument nx is invalid.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
nxd=nx
IF(ny<=-2 .OR. ny>=4) THEN
  PRINT *,'An error in SUBROUTINE pcspstn(nx,ny) of Plotter'
  PRINT *,'ny=',ny
  PRINT *,'The value of argument ny is invalid.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
nyd=ny
END SUBROUTINE pcspstn

SUBROUTINE pcsangle(theta)
! Section 2.23 of the manual "Computer Graphics PLOTTER"
! Setting the slant angle of character patterns.
USE plotter,ONLY: pi,theta_deg
IMPLICIT NONE
REAL,INTENT(IN):: theta
IF(ABS(theta)>2*pi) THEN
  PRINT *,'An error in SUBROUTINE pcsangle(theta) of Plotter'
  PRINT '(A,F6.1)',' theta=',theta
  PRINT *,'It is necessary that ABS(theta)<2*pi.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
theta_deg=theta*180/pi
END SUBROUTINE pcsangle

SUBROUTINE pcwrite(x1,y1,char)
! Section 2.24 of the manual "Computer Graphics PLOTTER"
! Writing a character pattern.
! The internal position of the character pattern is put at (x1, y1).
! The internal procedures: writeascii, shiftkanji, iachard, indexd and
! printchar.
USE plotter,ONLY: iopen,x00d,y00d,theta_deg,nxd,nyd,heightd,fact
IMPLICIT NONE
REAL,INTENT(IN):: x1,y1
CHARACTER(LEN=*),INTENT(IN):: char
CHARACTER(LEN=200):: char1,char20,char3
CHARACTER(LEN=3):: chsuf
CHARACTER(LEN=2):: chars
REAL,PARAMETER:: factspace=20     ! factspace=charsize/\s1}
REAL,PARAMETER:: factsubf=.7      ! factsubf=charsizesub/charsize
REAL:: a1,charsize_prnt,sp,sufpos,xshift,yshift,heightd1,charsize,charsizesub
INTEGER:: i1,i2,i3,i11,i12,isuf,numsuf,kindchar,ksp
REAL:: dcharfact(12)
DATA dcharfact/4*4.27,4*3.97,2*4.27,2*3.63/
CHARACTER(LEN=22):: chartype(12)
DATA chartype &
   /'Times-Roman','Times-Bold','Times-Italic','Times-BoldItalic',&
   'Helvetica','Helvetica-Bold','Helvetica-Oblique','Helvetica-BoldOblique', &
   'Symbol','Symbol','Ryumin-Light-H','GothicBBB-Medium-H'/
IF(iopen < 0) THEN
  PRINT *,'An error in SUBROUTINE pcwrite of Plotter'
  PRINT *,'SUBROUTINE pfnbegin must be invoked before SUBROUTINE pcwrite.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
! Making a scratch file
OPEN(UNIT=86,STATUS='SCRATCH')
REWIND 86
IF(LEN(char) > 200) THEN
  PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
  PRINT *,'LEN(char)=',LEN(char)
  PRINT *,'It is necessary that LEN(char)<=200.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
heightd1=heightd*fact  ! heightd1=the height of characters in unit pt.
kindchar=1
charsize=heightd*dcharfact(kindchar) ! =character size in scalefont
charsizesub=charsize*factsubf        ! =subscript size in scalefont
CALL printchar(charsize)
! Writing the control strings and the text strings of char to file unit 86.
char1=ADJUSTL(char)
DO
  i2=MIN(indexd(INDEX(char1,'\f')),indexd(INDEX(char1,'\u')),&
     indexd(INDEX(char1,'\l')),indexd(INDEX(char1,'\e')),&
     indexd(INDEX(char1,'\s')))  ! indexd: an internal function
  IF(i2 > 500) EXIT
  char20=char1(:i2-1);  i11=INDEX(char20,'\')
  DO
    IF(i11 == 0) EXIT
    IF(VERIFY(char20(i11+1:i11+3),'01234567') /= 0) THEN
      PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
      PRINT *,"char='",char,"'"
      PRINT *,'The letter next \ must be f, u, l, e, s or 3 octal figures.'
      PRINT *,'Stop of execution';  PRINT *;  STOP
    ENDIF
    i12=INDEX(char20(i11+1:),'\');  IF(i12 == 0) EXIT;  i11=i12+i11
  ENDDO
  IF(i2 >= 2) WRITE(86,102) char1(:i2-1)//'\'  ! letters to be printed
  char1=char1(i2:)
  i1=INDEX(char1,' ');  i3=INDEX(char1,'}')
  IF(i1 < i3) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'There is a space in a control string.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  IF(i3 == 0) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'Argument char lacks } in a control string.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  WRITE(86,102) char1(:i3-1)          !  Control string
  char1=char1(i3+1:)
ENDDO
i11=INDEX(char1,'\')
DO
  IF(i11 == 0) EXIT
  IF(VERIFY(char1(i11+1:i11+3),'01234567') /= 0) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'The letter next \ must be f, u, l, e, s or 3 octal figures.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  i12=INDEX(char1(i11+1:),'\');  IF(i12 == 0) EXIT;  i11=i12+i11
ENDDO
IF(LEN_TRIM(char1) >= 1) WRITE(86,102) TRIM(char1)//'\' ! Letters to be printed
ENDFILE 86;  REWIND 86
! Calculation of the length of the string = xxdis = a variable of PostScript
102 FORMAT(A,F9.2,A)
WRITE(88,102) '/xxdis 0 def '
isuf=0    ! isuf=0: outside subscripts,  isuf=1,2: inside subscripts
numsuf=0  ! numsuf: the total of subscripts.
DO
  READ(86,102,END=50) char3
  IF(iachard(char3(1:2)) == -1) THEN ! processing control strings
    SELECT CASE(char3(2:2))
      CASE('f')
        IF(LEN_TRIM(char3) <= 2) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is no figure after \f.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        IF(VERIFY(char3(3:),' 0123456789') /= 0) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is a letter other than figures after \f.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        READ(char3(3:),*) kindchar
        IF(kindchar <=0 .OR. kindchar >= 13) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'The font number is invalid.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        charsize=heightd*dcharfact(kindchar) ! =character size in scalefont
        charsizesub=charsize*factsubf        ! =sharsize of subscripts
        charsize_prnt=charsize
        IF(isuf >= 1) charsize_prnt=charsizesub
        CALL printchar(charsize_prnt)
      CASE('s')
        IF(LEN_TRIM(char3) <= 2) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is no figure after \s.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        IF(VERIFY(char3(3:),' +-0123456789') /= 0) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is a letter other than figures between \s and }.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        READ(char3(3:),*) ksp;  sp=(ksp/factspace)*heightd1
        SELECT CASE(isuf)
          CASE(0)
            WRITE(88,102) '/xxdis xxdis ',sp,' add def'
          CASE(1:2)
            WRITE(88,102) '/xxdis'//chsuf//' xxdis'//chsuf,sp,' add def'
          CASE DEFAULT;  STOP 'Error (1) in pcwrite of Plotter'
        END SELECT
      CASE('u','l');  isuf=isuf+1
        IF(isuf == 1) THEN;  numsuf=numsuf+1
          WRITE(chars,'(I2.2)') numsuf
          WRITE(88,*) '/xxdis'//chars//'1 0 def '
          WRITE(88,*) '/xxdis'//chars//'2 0 def '
        ENDIF
        WRITE(chsuf,'(I2.2,I1.1)') numsuf,isuf
        CALL printchar(charsizesub)
        IF(LEN_TRIM(char3) >= 3) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'The letter next \u or \l must be }.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
      CASE('e')
        IF(LEN_TRIM(char3) >= 3) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'The letter next \e must be }.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        CALL printchar(charsize)
        WRITE(chars,'(I2.2)') numsuf
        WRITE(88,*) ' xxdis'//chars//'1 xxdis'//chars//'2 2 copy lt &
             &{ exch } if pop xxdis add /xxdis exch def'
        IF(isuf <= 0) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is not \u} nor \l} before \e}.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        IF(isuf >= 3) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'The total number of \u} and \l} before \e}&
            & must be less than 3.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        isuf=0
      CASE DEFAULT;  STOP 'Error (2) in pcwrite of Plotter'
    END SELECT
  ELSE    ! Processing characters to be written
    i2=INDEX(char3,'\ ')
    IF(isuf == 0) THEN
      IF(kindchar <= 10) THEN
        WRITE(88,102) '/xxdis '
        CALL writeascii
        WRITE(88,102) ' stringwidth pop xxdis add def'
      ELSE
        IF(MOD(i2,2) == 0) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is an ASCII code in the Japanese sentence.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        WRITE(88,102) '/xxdis '
        CALL shiftkanji
        WRITE(88,102) ' stringwidth pop xxdis add def'
      ENDIF
    ELSE
      IF(kindchar <= 10) THEN
        CALL writeascii
        WRITE(88,102) ' stringwidth pop '
        WRITE(88,102) ' xxdis'//chsuf//' add /xxdis'//chsuf//' exch def'
      ELSE
        IF(MOD(i2,2) == 0) THEN
          PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
          PRINT *,"char='",char,"'"
          PRINT *,'There is an ASCII code in the Japanese sentence.'
          PRINT *,'Stop of execution';  PRINT *;  STOP
        ENDIF
        CALL shiftkanji
        WRITE(88,102) ' stringwidth pop '
        WRITE(88,102) ' xxdis'//chsuf//' add /xxdis'//chsuf//' exch def'
      ENDIF
    ENDIF
  ENDIF
ENDDO
50 CONTINUE
IF(isuf /= 0) THEN
  PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
  PRINT *,"char='",char,"'"
  PRINT *,'Argument char lacks the index control string \e}.'
  PRINT *,'Stop of execution';  PRINT *;  STOP
ENDIF
! Print of characters
REWIND 86;  isuf=0;  numsuf=0
WRITE(88,'(2F9.2,A)') fact*x1+x00d,fact*y1+y00d,' moveto'
WRITE(88,'(F9.2,A)') theta_deg,' rotate'
SELECT CASE(nxd)
  CASE(0);  xshift=0
  CASE(1);  xshift=-.5
  CASE(2);  xshift=-1
  CASE DEFAULT;  STOP 'Error (3) in pcwrite of Plotter'
END SELECT
WRITE(88,'(A,F9.2,A)') ' xxdis ',xshift,' mul '
SELECT CASE(nyd)
  CASE(-1);  yshift=  (7./21)*heightd1
  CASE( 0);  yshift=   0
  CASE( 1);  yshift= -(9./21)*heightd1
  CASE( 2);  yshift=         -heightd1
  CASE( 3);  yshift=-(25./21)*heightd1
  CASE DEFAULT;  STOP 'Error (4) in pcwrite of Plotter'
END SELECT
WRITE(88,'(F9.2,A)') yshift,' rmoveto'
kindchar=1
charsize=heightd*dcharfact(kindchar) !=character size in scalefont
charsizesub=charsize*factsubf        ! charsizesub: sharsize of subscripts
CALL printchar(charsize)
DO
  READ(86,102,END=60) char3
  IF(iachard(char3(1:2)) == -1) THEN   ! Processing control strings
    SELECT CASE(char3(2:2))
      CASE('f')
        READ(char3(3:),*) kindchar
        charsize=heightd*dcharfact(kindchar) !=character size in scalefont
        charsizesub=charsize*factsubf   ! charsizesub: sharsize of subscripts
        charsize_prnt=charsize
        IF(isuf >= 1) charsize_prnt=charsizesub
        CALL printchar(charsize_prnt)
      CASE('s')
        READ(char3(3:),*) ksp;  sp=(ksp/factspace)*heightd1
        WRITE(88,'(F9.2,A)') sp,' 0 rmoveto'
      CASE('u')
        IF(isuf == 0) THEN;  numsuf=numsuf+1
        WRITE(chars,'(I2.2)') numsuf;  ENDIF
        IF(isuf == 1) WRITE(88,102) ' xxdis'//chsuf//' neg 0  rmoveto'
        isuf=isuf+1;  sufpos=1
        WRITE(chsuf,'(I2.2,I1.1)') numsuf,isuf
        CALL printchar(charsizesub)
      CASE('l')
        IF(isuf == 0) THEN
          numsuf=numsuf+1;  WRITE(chars,'(I2.2)') numsuf
        ENDIF
        IF(isuf == 1) WRITE(88,102) ' xxdis'//chsuf//' neg 0  rmoveto'
        isuf=isuf+1;   sufpos=0
        WRITE(chsuf,'(I2.2,I1.1)') numsuf,isuf
        CALL printchar(charsizesub)
      CASE('e')
        CALL printchar(charsize)
        SELECT CASE(isuf)
          CASE(1:2)
            WRITE(88,*) ' xxdis'//chsuf//' neg 0  rmoveto'
            WRITE(88,*) ' xxdis'//chars//'1 xxdis'//chars//'2 2 copy lt &
              &{ exch } if pop 0 rmoveto'
          CASE DEFAULT;  STOP 'Error (5) in pcwrite of Plotter'
        END SELECT
        isuf=0
      CASE DEFAULT;  STOP 'Error (6) in pcwrite of Plotter'
    END SELECT
  ELSE ! processing characters to be printing
    IF(isuf == 0) THEN
      IF(kindchar <= 10) THEN
        CALL writeascii
        WRITE(88,102) ' show'
      ELSE
        CALL shiftkanji
        WRITE(88,102) ' show'
      ENDIF
    ELSE
      a1=(sufpos-0.3)*heightd1
      WRITE(88,102) ' 0 ',a1,' rmoveto'
      IF(kindchar <= 10) THEN
        CALL writeascii
        WRITE(88,102) ' show'
      ELSE
        CALL shiftkanji
        WRITE(88,102) ' show'
      ENDIF
      WRITE(88,102) ' 0 ',-a1,' rmoveto'
    ENDIF
  ENDIF
ENDDO
60  CONTINUE
WRITE(88,'(F9.2,A)') -theta_deg,' rotate '
CLOSE(UNIT=86,STATUS='DELETE')
CONTAINS

SUBROUTINE writeascii
! An internal procedure of SUBROUTINE pcwrite
! Including process of ) and (.
! The variable used in common with the host procedure: char,char3
INTEGER:: i6,ii,i2
CHARACTER(LEN=200):: char70
CHARACTER(LEN=2):: char2
i2=INDEX(char3,'\ ');  i6=1
DO ii=1,i2-1
  WRITE(char2,'(Z2)') ICHAR(char3(ii:ii))
  IF(.NOT.(('20'<=char2 .AND. char2<='7E') .OR. &
    ('A1'<=char2 .AND. char2<='FE'))) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'There is a non-ASCII code in the English sentence.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  SELECT CASE(char3(ii:ii))
    CASE('(')
      char70(i6:)='\050'//char3(ii+1:)
      i6=i6+4
    CASE(')')
      char70(i6:)='\051'//char3(ii+1:)
      i6=i6+4
    CASE DEFAULT
      char70(i6:i6)=char3(ii:ii)
      i6=i6+1
  END SELECT
ENDDO
WRITE(88,102) '('//char70(:i6-1)//')'
102 FORMAT(A,F9.2,A)
END SUBROUTINE writeascii

SUBROUTINE shiftkanji
! An internal procedure of SUBROUTINE pcwrite
! Processing kanji
! Conversion from a shift JIS code into the JIS code.
! Ensaikuropedia jyohou shori p.474
! The variable used in common with the host procedure: char3
INTEGER:: id(200),il,io,iach1,i1,i2,ik,is1,is2,j1,i,j2
CHARACTER(LEN=300):: char300
il=0
DO
  io=il*2+1
  iach1=iachard(char3(io:io+1))
  IF(iach1 == -1) EXIT
  IF(iach1 == -2) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'There is an ASCII code in the Japanese sentence.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  i2=MOD(iach1,256);  i1=iach1/256
  ik=224;  IF(i1 >= 220) ik=352
  is1=0;  is2=0
  IF(i2 <= 158) THEN;  is1=1;  is2=95;  ENDIF
  j1=2*i1-ik-is1;  j2=i2-126+is2
  IF(i2 <= 158 .AND. j2 >= 96) j2=j2-1
  IF(.NOT.((j1>=33 .AND. j1<=116) .AND. (j2>=33 .AND. j2<=126))) THEN
    PRINT *,'An error in SUBROUTINE pcwrite(x1,y1,char) of Plotter'
    PRINT *,"char='",char,"'"
    PRINT *,'There is a non-kanji code in the Japanese sentence.'
    PRINT *,'Stop of execution';  PRINT *;  STOP
  ENDIF
  il=il+1
  id(il)=j1*256+j2  !  16*16=256
ENDDO
WRITE(char300,'(200Z5)') (id(i),i=1,il)
WRITE(88,'(A)') '<'//TRIM(char300)//'>'
END SUBROUTINE shiftkanji

FUNCTION iachard(chrr2) RESULT(iachart)
! An internal procedure of SUBROUTINE pcwrite
! Conversion from a hexadecimal number to the decimal number.
! The variable used in common with the host procedure: none
CHARACTER(LEN=2),INTENT(IN):: chrr2
INTEGER:: iachart,ik1,ik2
CHARACTER(LEN=4):: chrr4
WRITE(chrr4,'(Z4)') ICHAR(chrr2(1:1))*256+ICHAR(chrr2(2:2))
READ(chrr4(1:2),'(Z2)') ik1
IF(ik1 == 92 .AND. SCAN('fules ',chrr2(2:2)) >= 1) THEN   ! ik1=92 for '\'
  iachart=-1;  RETURN   ! a control string
ENDIF
READ(chrr4(3:4),'(Z2)') ik2
IF(ik2 == 92) THEN;  iachart=-2;  RETURN;  ENDIF          ! ik2=92 for '\'
iachart=ik1*256+ik2  !  16*16=256
END FUNCTION iachard

FUNCTION indexd(i) RESULT(indext)
! An internal procedure of SUBROUTINE pcwrite
INTEGER,INTENT(IN):: i
INTEGER:: indext
IF(i == 0) THEN; indext=550;  ELSE;  indext=i;  ENDIF
END FUNCTION indexd

SUBROUTINE printchar(charsized)
! An internal procedure of SUBROUTINE pcwrite
! Printing a size and a font of characters.
! The variables used in common with the host procedure: chartype and kindchar
REAL,INTENT(IN):: charsized
IF(kindchar /= 10) THEN
  WRITE(88,102) '/'//chartype(kindchar)//' findfont ',charsized, &
         ' scalefont setfont '
  102 FORMAT(A,F9.2,A)
ELSE
  WRITE(88,105) '/'//chartype(kindchar),' findfont [ ',charsized, &
        ' 0 ',charsized*.25,charsized,-.045*charsized,' 0 ] makefont setfont '
  105 FORMAT(A/A,1F9.2,A,3F9.2,A)
ENDIF
END SUBROUTINE printchar

END SUBROUTINE pcwrite

