      SUBROUTINE DSYMM( SIDE, UPLO, M, N, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        SIDE, UPLO
      INTEGER            M, N, LDA, LDB, LDC
      DOUBLE PRECISION   ALPHA, BETA
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), B( LDB, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  DSYMM  performs one of the matrix-matrix operations
*
*     C := alpha*A*B + beta*C,
*
*  or
*
*     C := alpha*B*A + beta*C,
*
*  where alpha and beta are scalars,  A is a symmetric matrix and  B and
*  C are  m by n matrices.
*
*  Parameters
*  ==========
*
*  SIDE   - CHARACTER*1.
*           On entry,  SIDE  specifies whether  the  symmetric matrix  A
*           appears on the  left or right  in the  operation as follows:
*
*              SIDE = 'L' or 'l'   C := alpha*A*B + beta*C,
*
*              SIDE = 'R' or 'r'   C := alpha*B*A + beta*C,
*
*           Unchanged on exit.
*
*  UPLO   - CHARACTER*1.
*           On  entry,   UPLO  specifies  whether  the  upper  or  lower
*           triangular  part  of  the  symmetric  matrix   A  is  to  be
*           referenced as follows:
*
*              UPLO = 'U' or 'u'   Only the upper triangular part of the
*                                  symmetric matrix is to be referenced.
*
*              UPLO = 'L' or 'l'   Only the lower triangular part of the
*                                  symmetric matrix is to be referenced.
*
*           Unchanged on exit.
*
*  M      - INTEGER.
*           On entry,  M  specifies the number of rows of the matrix  C.
*           M  must be at least zero.
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry, N specifies the number of columns of the matrix C.
*           N  must be at least zero.
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry, ALPHA specifies the scalar alpha.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
*           m  when  SIDE = 'L' or 'l'  and is  n otherwise.
*           Before entry  with  SIDE = 'L' or 'l',  the  m by m  part of
*           the array  A  must contain the  symmetric matrix,  such that
*           when  UPLO = 'U' or 'u', the leading m by m upper triangular
*           part of the array  A  must contain the upper triangular part
*           of the  symmetric matrix and the  strictly  lower triangular
*           part of  A  is not referenced,  and when  UPLO = 'L' or 'l',
*           the leading  m by m  lower triangular part  of the  array  A
*           must  contain  the  lower triangular part  of the  symmetric
*           matrix and the  strictly upper triangular part of  A  is not
*           referenced.
*           Before entry  with  SIDE = 'R' or 'r',  the  n by n  part of
*           the array  A  must contain the  symmetric matrix,  such that
*           when  UPLO = 'U' or 'u', the leading n by n upper triangular
*           part of the array  A  must contain the upper triangular part
*           of the  symmetric matrix and the  strictly  lower triangular
*           part of  A  is not referenced,  and when  UPLO = 'L' or 'l',
*           the leading  n by n  lower triangular part  of the  array  A
*           must  contain  the  lower triangular part  of the  symmetric
*           matrix and the  strictly upper triangular part of  A  is not
*           referenced.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
*           LDA must be at least  max( 1, m ), otherwise  LDA must be at
*           least  max( 1, n ).
*           Unchanged on exit.
*
*  B      - DOUBLE PRECISION array of DIMENSION ( LDB, n ).
*           Before entry, the leading  m by n part of the array  B  must
*           contain the matrix B.
*           Unchanged on exit.
*
*  LDB    - INTEGER.
*           On entry, LDB specifies the first dimension of B as declared
*           in  the  calling  (sub)  program.   LDB  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*
*  BETA   - DOUBLE PRECISION.
*           On entry,  BETA  specifies the scalar  beta.  When  BETA  is
*           supplied as zero then C need not be set on input.
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry, the leading  m by n  part of the array  C must
*           contain the matrix  C,  except when  beta  is zero, in which
*           case C need not be set on entry.
*           On exit, the array  C  is overwritten by the  m by n updated
*           matrix.
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*
*
*  Level 3 Blas routine.
*
*  -- Written on 8-February-1989.
*     Jack Dongarra, Argonne National Laboratory.
*     Iain Duff, AERE Harwell.
*     Jeremy Du Croz, Numerical Algorithms Group Ltd.
*     Sven Hammarling, Numerical Algorithms Group Ltd.
*
*  -- Rewritten in December-1993.
*     GEMM-Based Level 3 BLAS.
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Rewritten in Mars-1995.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Department of Computing Science,
*     University of Umea, Sweden.
*
*
*     .. Local Scalars ..
      INTEGER            INFO, NROWA
      LOGICAL            LSIDE, UPPER
      INTEGER            I, J, II, IX, ISEC, UISEC
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN, MOD
*     .. External Functions ..
      LOGICAL            LSAME
      EXTERNAL           LSAME
*     .. External Subroutines ..
      EXTERNAL           XERBLA
      EXTERNAL           DGEMM
*     .. Parameters ..
      DOUBLE PRECISION   ZERO, ONE
      PARAMETER        ( ZERO = 0.0D+0, ONE = 1.0D+0 )
*     .. User specified parameters for DSYMM ..
      INTEGER            RCB
      PARAMETER        ( RCB = 96 )
*     .. Local Arrays ..
      DOUBLE PRECISION   T1( RCB, RCB )
*     ..
*     .. Executable Statements ..
*
*     Test the input parameters.
*
      LSIDE = LSAME( SIDE, 'L' )
      UPPER = LSAME( UPLO, 'U' )
      IF( LSIDE )THEN
         NROWA = M
      ELSE
         NROWA = N
      END IF
      INFO = 0
      IF( ( .NOT.LSIDE ).AND.( .NOT.LSAME( SIDE, 'R' ) ) )THEN
         INFO = 1
      ELSE IF( ( .NOT.UPPER ).AND.( .NOT.LSAME( UPLO, 'L' ) ) )THEN
         INFO = 2
      ELSE IF( M.LT.0 )THEN
         INFO = 3
      ELSE IF( N.LT.0 )THEN
         INFO = 4
      ELSE IF( LDA.LT.MAX( 1, NROWA ) )THEN
         INFO = 7
      ELSE IF( LDB.LT.MAX( 1, M ) )THEN
         INFO = 9
      ELSE IF( LDC.LT.MAX( 1, M ) )THEN
         INFO = 12
      END IF
      IF( INFO.NE.0 )THEN
         CALL XERBLA( 'DSYMM ', INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( ( M.EQ.0 ).OR.( N.EQ.0 ).OR.
     $    ( ( ALPHA.EQ.ZERO ).AND.( BETA.EQ.ONE ) ) )
     $   RETURN
*
*     And when alpha.eq.zero.
*
      IF( ALPHA.EQ.ZERO )THEN
         IF( BETA.EQ.ZERO )THEN
            UISEC = M-MOD( M, 4 )
            DO 30, J = 1, N
               DO 10, I = 1, UISEC, 4
                  C( I, J ) = ZERO
                  C( I+1, J ) = ZERO
                  C( I+2, J ) = ZERO
                  C( I+3, J ) = ZERO
   10          CONTINUE
               DO 20, I = UISEC+1, M
                  C( I, J ) = ZERO
   20          CONTINUE
   30       CONTINUE
         ELSE
            UISEC = M-MOD( M, 4 )
            DO 60, J = 1, N
               DO 40, I = 1, UISEC, 4
                  C( I, J ) = BETA*C( I, J )
                  C( I+1, J ) = BETA*C( I+1, J )
                  C( I+2, J ) = BETA*C( I+2, J )
                  C( I+3, J ) = BETA*C( I+3, J )
   40          CONTINUE
               DO 50, I = UISEC+1, M
                  C( I, J ) = BETA*C( I, J )
   50          CONTINUE
   60       CONTINUE
         END IF
         RETURN
      END IF
*
*     Start the operations.
*
      IF( LSIDE )THEN
         IF( UPPER )THEN
*
*           Form  C := alpha*A*B + beta*C. Left, Upper.
*
            DO 90, II = 1, M, RCB
               ISEC = MIN( RCB, M-II+1 )
*
*              T1 := A, the upper triangular part of a square diagonal
*              block of A is copied to upper pangular part of T1 and
*              the transpose of the strictly upper triangular part of
*              the block of A is copied to the strictly lower
*              triangular part of T1.
*
               DO 80, J = II+ISEC-2, II-1, -2
                  UISEC = J-II-MOD( J-II, 2 )
                  DO 70, I = II, II+UISEC-1, 2
                     T1( I-II+1, J-II+1 ) = ALPHA*A( I, J )
                     T1( I-II+2, J-II+1 ) = ALPHA*A( I+1, J )
                     T1( I-II+1, J-II+2 ) = ALPHA*A( I, J+1 )
                     T1( I-II+2, J-II+2 ) = ALPHA*A( I+1, J+1 )
                     T1( J-II+1, I-II+1 ) = ALPHA*A( I, J )
                     T1( J-II+1, I-II+2 ) = ALPHA*A( I+1, J )
                     T1( J-II+2, I-II+1 ) = ALPHA*A( I, J+1 )
                     T1( J-II+2, I-II+2 ) = ALPHA*A( I+1, J+1 )
   70             CONTINUE
                  IF( MOD( J-II, 2 ).EQ.1 )THEN
                     T1( J-II, J-II+1 ) = ALPHA*A( J-1, J )
                     T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                     T1( J-II, J-II+2 ) = ALPHA*A( J-1, J+1 )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J, J+1 )
                     T1( J-II+1, J-II ) = ALPHA*A( J-1, J )
                     T1( J-II+2, J-II ) = ALPHA*A( J-1, J+1 )
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J, J+1 )
                  ELSE IF( J.GE.II )THEN
                     T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J, J+1 )
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J, J+1 )
                  END IF
                  T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
   80          CONTINUE
*
*              C := T1'*B + beta*C, general matrix multiplication
*              involving the symmetric diagonal block of A stored
*              as a full matrix block in T1.
*
               CALL DGEMM ( 'T', 'N', ISEC, N, ISEC, ONE,
     $                                 T1( 1, 1 ), RCB, B( II, 1 ), LDB,
     $                                           BETA, C( II, 1 ), LDC )
               IF( II.GT.1 )THEN
*
*                 C := alpha*A'*B + C, general matrix multiplication
*                 involving the transpose of a rectangular block of A.
*
                  CALL DGEMM ( 'T', 'N', ISEC, N, II-1, ALPHA,
     $                                  A( 1, II ), LDA, B( 1, 1 ), LDB,
     $                                            ONE, C( II, 1 ), LDC )
               END IF
               IF( II+ISEC.LE.M )THEN
*
*                 C := alpha*A*B + C, general matrix multiplication
*                 involving a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'N', ISEC, N, M-II-ISEC+1, ALPHA,
     $                      A( II, II+ISEC ), LDA, B( II+ISEC, 1 ), LDB,
     $                                            ONE, C( II, 1 ), LDC )
               END IF
   90       CONTINUE
         ELSE
*
*           Form  C := alpha*A*B + beta*C. Left, Lower.
*
            DO 120, IX = M, 1, -RCB
               II = MAX( 1, IX-RCB+1 )
               ISEC = IX-II+1
*
*              T1 := A, the lower triangular part of a square diagonal
*              block of A is copied to lower pangular part of T1 and
*              the transpose of the strictly lower triangular part of
*              the block of A is copied to the strictly upper
*              triangular part of T1.
*
               DO 110, J = II, II+ISEC-1, 2
                  UISEC = II+ISEC-J-2-MOD( II+ISEC-J-2, 2 )
                  T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                  IF( MOD( II+ISEC-J-2, 2 ).EQ.0 )THEN
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J+1, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J+1, J )
                     T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
                  ELSE IF( J.LE.II+ISEC-3 )THEN
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J+1, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J+1, J )
                     T1( J-II+3, J-II+1 ) = ALPHA*A( J+2, J )
                     T1( J-II+1, J-II+3 ) = ALPHA*A( J+2, J )
                     T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
                     T1( J-II+3, J-II+2 ) = ALPHA*A( J+2, J+1 )
                     T1( J-II+2, J-II+3 ) = ALPHA*A( J+2, J+1 )
                  END IF
                  DO 100 I = II+ISEC-UISEC, II+ISEC-1, 2
                     T1( I-II+1, J-II+1 ) = ALPHA*A( I, J )
                     T1( I-II+2, J-II+1 ) = ALPHA*A( I+1, J )
                     T1( I-II+1, J-II+2 ) = ALPHA*A( I, J+1 )
                     T1( I-II+2, J-II+2 ) = ALPHA*A( I+1, J+1 )
                     T1( J-II+1, I-II+1 ) = ALPHA*A( I, J )
                     T1( J-II+1, I-II+2 ) = ALPHA*A( I+1, J )
                     T1( J-II+2, I-II+1 ) = ALPHA*A( I, J+1 )
                     T1( J-II+2, I-II+2 ) = ALPHA*A( I+1, J+1 )
  100             CONTINUE
  110          CONTINUE
*
*              C := T1'*B + beta*C, general matrix multiplication
*              involving the symmetric diagonal block of A stored
*              as a full matrix block in T1.
*
               CALL DGEMM ( 'T', 'N', ISEC, N, ISEC, ONE,
     $                                 T1( 1, 1 ), RCB, B( II, 1 ), LDB,
     $                                           BETA, C( II, 1 ), LDC )
               IF( II.GT.1 )THEN
*
*                 C := alpha*A'*B + C, general matrix multiplication
*                 involving the transpose of a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'N', ISEC, N, II-1, ALPHA,
     $                                  A( II, 1 ), LDA, B( 1, 1 ), LDB,
     $                                            ONE, C( II, 1 ), LDC )
               END IF
               IF( II+ISEC.LE.M )THEN
*
*                 C := alpha*A*B + C, general matrix multiplication
*                 involving a rectangular block of A.
*
                  CALL DGEMM ( 'T', 'N', ISEC, N, M-II-ISEC+1, ALPHA,
     $                      A( II+ISEC, II ), LDA, B( II+ISEC, 1 ), LDB,
     $                                            ONE, C( II, 1 ), LDC )
               END IF
  120       CONTINUE
         END IF
      ELSE
         IF( UPPER )THEN
*
*           Form  C := alpha*B*A + beta*C. Right, Upper.
*
            DO 150, II = 1, N, RCB
               ISEC = MIN( RCB, N-II+1 )
*
*              T1 := A, the upper triangular part of a square diagonal
*              block of A is copied to upper pangular part of T1 and
*              the transpose of the strictly upper triangular part of
*              the block of A is copied to the strictly lower
*              triangular part of T1.
*
               DO 140, J = II+ISEC-2, II-1, -2
                  UISEC = J-II-MOD( J-II, 2 )
                  DO 130, I = II, II+UISEC-1, 2
                     T1( I-II+1, J-II+1 ) = ALPHA*A( I, J )
                     T1( I-II+2, J-II+1 ) = ALPHA*A( I+1, J )
                     T1( I-II+1, J-II+2 ) = ALPHA*A( I, J+1 )
                     T1( I-II+2, J-II+2 ) = ALPHA*A( I+1, J+1 )
                     T1( J-II+1, I-II+1 ) = ALPHA*A( I, J )
                     T1( J-II+1, I-II+2 ) = ALPHA*A( I+1, J )
                     T1( J-II+2, I-II+1 ) = ALPHA*A( I, J+1 )
                     T1( J-II+2, I-II+2 ) = ALPHA*A( I+1, J+1 )
  130             CONTINUE
                  IF( MOD( J-II, 2 ).EQ.1 )THEN
                     T1( J-II, J-II+1 ) = ALPHA*A( J-1, J )
                     T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                     T1( J-II, J-II+2 ) = ALPHA*A( J-1, J+1 )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J, J+1 )
                     T1( J-II+1, J-II ) = ALPHA*A( J-1, J )
                     T1( J-II+2, J-II ) = ALPHA*A( J-1, J+1 )
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J, J+1 )
                  ELSE IF( J.GE.II )THEN
                     T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J, J+1 )
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J, J+1 )
                  END IF
                  T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
  140          CONTINUE
*
*              C := T1'*B + beta*C, general matrix multiplication
*              involving the symmetric diagonal block of A stored
*              as a full matrix block in T1.
*
               CALL DGEMM ( 'N', 'N', M, ISEC, ISEC, ONE,
     $                                 B( 1, II ), LDB, T1( 1, 1 ), RCB,
     $                                           BETA, C( 1, II ), LDC )
               IF( II.GT.1 )THEN
*
*                 C := alpha*B*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'N', M, ISEC, II-1, ALPHA,
     $                                  B( 1, 1 ), LDB, A( 1, II ), LDA,
     $                                            ONE, C( 1, II ), LDC )
               END IF
               IF( II+ISEC.LE.N )THEN
*
*                 C := alpha*B*A' + C, general matrix multiply involving
*                 the transpose of a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'T', M, ISEC, N-II-ISEC+1, ALPHA,
     $                      B( 1, II+ISEC ), LDB, A( II, II+ISEC ), LDA,
     $                                            ONE, C( 1, II ), LDC )
               END IF
  150       CONTINUE
         ELSE
*
*           Form  C := alpha*B*A + beta*C. Right, Lower.
*
            DO 180, IX = N, 1, -RCB
               II = MAX( 1, IX-RCB+1 )
               ISEC = IX-II+1
*
*              T1 := A, the lower triangular part of a square diagonal
*              block of A is copied to lower pangular part of T1 and
*              the transpose of the strictly lower triangular part of
*              the block of A is copied to the strictly upper
*              triangular part of T1.
*
               DO 170, J = II, II+ISEC-1, 2
                  UISEC = II+ISEC-J-2-MOD( II+ISEC-J-2, 2 )
                  T1( J-II+1, J-II+1 ) = ALPHA*A( J, J )
                  IF( MOD( II+ISEC-J-2, 2 ).EQ.0 )THEN
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J+1, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J+1, J )
                     T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
                  ELSE IF( J.LE.II+ISEC-3 )THEN
                     T1( J-II+2, J-II+1 ) = ALPHA*A( J+1, J )
                     T1( J-II+1, J-II+2 ) = ALPHA*A( J+1, J )
                     T1( J-II+3, J-II+1 ) = ALPHA*A( J+2, J )
                     T1( J-II+1, J-II+3 ) = ALPHA*A( J+2, J )
                     T1( J-II+2, J-II+2 ) = ALPHA*A( J+1, J+1 )
                     T1( J-II+3, J-II+2 ) = ALPHA*A( J+2, J+1 )
                     T1( J-II+2, J-II+3 ) = ALPHA*A( J+2, J+1 )
                  END IF
                  DO 160 I = II+ISEC-UISEC, II+ISEC-1, 2
                     T1( I-II+1, J-II+1 ) = ALPHA*A( I, J )
                     T1( I-II+2, J-II+1 ) = ALPHA*A( I+1, J )
                     T1( I-II+1, J-II+2 ) = ALPHA*A( I, J+1 )
                     T1( I-II+2, J-II+2 ) = ALPHA*A( I+1, J+1 )
                     T1( J-II+1, I-II+1 ) = ALPHA*A( I, J )
                     T1( J-II+1, I-II+2 ) = ALPHA*A( I+1, J )
                     T1( J-II+2, I-II+1 ) = ALPHA*A( I, J+1 )
                     T1( J-II+2, I-II+2 ) = ALPHA*A( I+1, J+1 )
  160             CONTINUE
  170          CONTINUE
*
*              C := T1'*B + beta*C, general matrix multiplication
*              involving the symmetric diagonal block of A stored
*              as a full matrix block in T1.
*
               CALL DGEMM ( 'N', 'N', M, ISEC, ISEC, ONE,
     $                                 B( 1, II ), LDB, T1( 1, 1 ), RCB,
     $                                           BETA, C( 1, II ), LDC )
               IF( II.GT.1 )THEN
*
*                 C := alpha*B*A' + C, general matrix multiply involving
*                 the transpose of a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'T', M, ISEC, II-1, ALPHA,
     $                                  B( 1, 1 ), LDB, A( II, 1 ), LDA,
     $                                            ONE, C( 1, II ), LDC )
               END IF
               IF( II+ISEC.LE.N )THEN
*
*                 C := alpha*B*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  CALL DGEMM ( 'N', 'N', M, ISEC, N-II-ISEC+1, ALPHA,
     $                      B( 1, II+ISEC ), LDB, A( II+ISEC, II ), LDA,
     $                                            ONE, C( 1, II ), LDC )
               END IF
  180       CONTINUE
         END IF
      END IF
*
      RETURN
*
*     End of DSYMM.
*
      END
